/**
 * @file EvolutionGeniviDbusHfpNetworkOperatorRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS Proxy for HfpNetworkOperator
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpNetworkOperatorRequestIf.h"
#include "IEvolutionGeniviDbusHfpNetworkOperatorCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::NetworkOperator;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpNetworkOperatorRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpNetworkOperatorRequestIf::EvolutionGeniviDbusHfpNetworkOperatorRequestIf()
{
   // _hfpNetworkOperatorProxy is set later
   _callbackIf = nullptr;
}

EvolutionGeniviDbusHfpNetworkOperatorRequestIf::~EvolutionGeniviDbusHfpNetworkOperatorRequestIf()
{
   _callbackIf = nullptr;
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR3(("destroyProxy(): NetworkOperator"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpNetworkOperatorProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("createRuntimeProxy(): NetworkOperator: ObjPath: %s", objPath.c_str()));

   if(nullptr != _callbackIf)
   {
      if(false == _hfpNetworkOperatorProxy.isInstanceAvailable(objPath))
      {
         ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
         ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connectorOptions;
         ::ccdbusif::convertBusType2ConnectorOption(connectorOptions,
               parser.getInterface2BusType(::ccdbusif::evolution::IF_NETWORK_OPERATOR));

         ::boost::shared_ptr< NetworkOperatorProxy > hfpNetworkOperatorProxy = NetworkOperatorProxy::createProxy(
               "pmAppEvoHfpNetworkOperatorPort",
               parser.getInterface2BusName(::ccdbusif::evolution::IF_NETWORK_OPERATOR),
               objPath,
               connectorOptions,
               *this);

         _hfpNetworkOperatorProxy.addProxyInstance(objPath, hfpNetworkOperatorProxy);
      }
      else
      {
         ETG_TRACE_ERR(("NetworkOperator Proxy already existing for requested ObjPath"));
      }
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("destroyRuntimeProxy(): NetworkOperator: ObjPath: %s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally;
   //       therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration
   //       (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< NetworkOperatorProxy > hfpNetworkOperatorProxy;
   if(true == _hfpNetworkOperatorProxy.isProxyAvailable(hfpNetworkOperatorProxy, objPath))
   {
      hfpNetworkOperatorProxy->sendDeregisterAll();
   }
   else
   {
      ETG_TRACE_ERR(("NetworkOperator Proxy not exist for requested ObjPath"));
   }

   _hfpNetworkOperatorProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::setCallbackIf(
      IEvolutionGeniviDbusHfpNetworkOperatorCallBackIf* callbackIf)
{
   _callbackIf = callbackIf;

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::networkOperatorCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if(nullptr != _callbackIf)
   {
      createRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::networkOperatorRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if(nullptr != _callbackIf)
   {
      destroyRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onAvailable(): NetworkOperator"));

   for(unsigned int i = 0; i < _hfpNetworkOperatorProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< NetworkOperatorProxy >& hfpNetworkOperatorProxy = _hfpNetworkOperatorProxy[i];
      if((nullptr != hfpNetworkOperatorProxy.get()) && (proxy == hfpNetworkOperatorProxy))
      {
         if(nullptr != _callbackIf)
         {
            hfpNetworkOperatorProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpNetworkOperatorProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onUnavailable(): NetworkOperator"));

   for(unsigned int i = 0; i < _hfpNetworkOperatorProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< NetworkOperatorProxy >& hfpNetworkOperatorProxy = _hfpNetworkOperatorProxy[i];
      if((nullptr != hfpNetworkOperatorProxy.get()) && (proxy == hfpNetworkOperatorProxy))
      {
         hfpNetworkOperatorProxy->sendDeregisterAll();

         if(nullptr != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpNetworkOperatorProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

act_t EvolutionGeniviDbusHfpNetworkOperatorRequestIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< NetworkOperatorProxy > proxy;

   if(true == _hfpNetworkOperatorProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }

   return act;
}

} //evobtstackwrapper
