/**
 * @file EvolutionGeniviDbusHfpModemRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for EvolutionGenivi DBUS Proxy for HfpModem
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup BtStackWrapper
 */

#include "EvolutionGeniviDbusHfpModemRequestIf.h"
#include "IEvolutionGeniviDbusHfpModemCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "EvolutionGeniviDbusParser.h"
#include "PmAppTrace.h"

using namespace ::org::ofono::Modem;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpModemRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BT_STACK_WRAPPER
#endif
#endif

namespace evobtstackwrapper {

EvolutionGeniviDbusHfpModemRequestIf::EvolutionGeniviDbusHfpModemRequestIf()
{
   // _hfpModemProxy is set later
   _callbackIf = nullptr;
}

EvolutionGeniviDbusHfpModemRequestIf::~EvolutionGeniviDbusHfpModemRequestIf()
{
   _callbackIf = nullptr;
}

void EvolutionGeniviDbusHfpModemRequestIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpModemRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR3(("destroyProxy(): Modem"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpModemProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpModemRequestIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("createRuntimeProxy(): Modem: ObjPath: %s", objPath.c_str()));

   if(nullptr != _callbackIf)
   {
      if(false == _hfpModemProxy.isInstanceAvailable(objPath))
      {
         ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
         ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connectorOptions;
         ::ccdbusif::convertBusType2ConnectorOption(connectorOptions,
               parser.getInterface2BusType(::ccdbusif::evolution::IF_MODEM));

         ::boost::shared_ptr< ModemProxy > hfpModemProxy = ModemProxy::createProxy(
               "pmAppEvoHfpModemPort",
               parser.getInterface2BusName(::ccdbusif::evolution::IF_MODEM),
               objPath,
               connectorOptions,
               *this);

         _hfpModemProxy.addProxyInstance(objPath, hfpModemProxy);
      }
      else
      {
         ETG_TRACE_ERR(("Modem Proxy already existing for requested ObjPath"));
      }
   }
}

void EvolutionGeniviDbusHfpModemRequestIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3(("destroyRuntimeProxy(): Modem: ObjPath: %s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally;
   //       therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration
   //       (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< ModemProxy > hfpModemProxy;
   if(true == _hfpModemProxy.isProxyAvailable(hfpModemProxy, objPath))
   {
      hfpModemProxy->sendDeregisterAll();
   }
   else
   {
      ETG_TRACE_ERR(("Modem Proxy not exist for requested ObjPath"));
   }

   _hfpModemProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpModemRequestIf::setCallbackIf(IEvolutionGeniviDbusHfpModemCallBackIf* callbackIf)
{
   _callbackIf = callbackIf;

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpModemRequestIf::modemCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if(nullptr != _callbackIf)
   {
      createRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpModemRequestIf::modemRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if(nullptr != _callbackIf)
   {
      destroyRuntimeProxy(objPath);
   }
}

void EvolutionGeniviDbusHfpModemRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onAvailable(): Modem"));

   for(unsigned int i = 0; i < _hfpModemProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ModemProxy >& hfpModemProxy = _hfpModemProxy[i];
      if((nullptr != hfpModemProxy.get()) && (proxy == hfpModemProxy))
      {
         if(nullptr != _callbackIf)
         {
            hfpModemProxy->sendPropertyChangedRegister(*_callbackIf);
            hfpModemProxy->sendSCOConnectRequestRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpModemProxy->getDBusObjectPath());

            //GetProperties
            sendGetPropertiesRequest(hfpModemProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpModemRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy,
      const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3(("onUnavailable(): Modem"));

   for(unsigned int i = 0; i < _hfpModemProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ModemProxy >& hfpModemProxy = _hfpModemProxy[i];
      if((nullptr != hfpModemProxy.get()) && (proxy == hfpModemProxy))
      {
         hfpModemProxy->sendDeregisterAll();

         if(nullptr != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
                  hfpModemProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

act_t EvolutionGeniviDbusHfpModemRequestIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< ModemProxy > proxy;

   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpModemRequestIf::sendSetPropertyRequest(const ::std::string& objPath,
      const ::std::string& property, const bool value)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< ModemProxy > proxy;

   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         ::asf::dbus::DBusVariant variant;
         ::ccdbusif::DbusVariant vValue;
         vValue.setBool(value);
         ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;
         parser.addVariantValue(variant, vValue);

         act = proxy->sendSetPropertyRequest(*_callbackIf, property, variant);
      }
   }

   return act;
}

act_t EvolutionGeniviDbusHfpModemRequestIf::sendAcceptSCOConnectRequest(const ::std::string& objPath)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;
   ::boost::shared_ptr< ModemProxy > proxy;

   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((nullptr != _callbackIf) && (nullptr != proxy.get()))
      {
         act = proxy->sendAcceptSCOConnectRequest(*_callbackIf);
      }
   }

   return act;
}

} //evobtstackwrapper
