/**
 * @file FileUtils.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of file utility functions used by PhoneCallManager
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details.
 *
 * @ingroup PmCommon
 */

#include "FileUtils.h"
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FileUtils.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace com
{
namespace bosch
{
namespace pmcommon
{

bool isDirectory(const std::string& dirPath)
{
   ETG_TRACE_USR1(("isDirectory() DirPath : %s", dirPath.c_str()));

   bool isDir = false;

   if(false == dirPath.empty())
   {
      struct stat statBuf;
      if ((stat(dirPath.c_str(), &statBuf) == 0) && (((statBuf.st_mode) & S_IFMT) == S_IFDIR))
      {
         // Directory is present
         isDir = true;
      }
   }

   ETG_TRACE_USR1(("isDirectory() isDir : %d", isDir));

   return isDir;
}

bool checkDirectoryExist(const std::string& dirPath)
{
   ETG_TRACE_USR1(("checkDirectoryExist() DirPath : %s", dirPath.c_str()));

   bool dirExist = false;

   if(false == dirPath.empty())
   {
      struct stat statBuf;
      if ((stat(dirPath.c_str(), &statBuf) == 0) && (((statBuf.st_mode) & S_IFMT) == S_IFDIR))
      {
         // Directory is present
         dirExist = true;
      }
   }

   ETG_TRACE_USR1(("isDirectory() dirExist : %d", dirExist));

   return dirExist;
}

int createDirectory(const std::string& dirPath, const unsigned int dirPermissions)
{
   ETG_TRACE_USR1(("createDirectory() DirPath : %s", dirPath.c_str()));

   int result = -1;

   if(false == dirPath.empty())
   {
      if(false == checkDirectoryExist(dirPath))
      {
         result = mkdir(dirPath.c_str(), dirPermissions);
         if (-1 == result)
         {
            ETG_TRACE_USR1(("createDirectory() failed"));
         }
         else
         {
            ETG_TRACE_USR1(("createDirectory() successful"));
         }
      }
      else
      {
         //Directory already exists
         result = 0;
         ETG_TRACE_USR1(("createDirectory() already exists"));
      }
   }

   return result;
}

bool checkFileExist(const std::string& filePath)
{
   ETG_TRACE_USR1(("checkFileExist() FilePath : %s", filePath.c_str()));

   bool fileExist = false;

   if(false == filePath.empty())
   {
      if((access(filePath.c_str(), F_OK)) != -1)
      {
         // File is present in the path
         fileExist = true;
      }
   }

   ETG_TRACE_USR1(("checkFileExist() fileExist : %d", fileExist));

   return fileExist;
}

int checkFilePermission(const std::string& filePath)
{
   ETG_TRACE_USR1(("checkFilePermission() FilePath : %s", filePath.c_str()));

   int result = 0;
   struct stat oStatBuf;

   // Retrieve current "owner", "group" and "permissions" of database file
   result = stat(filePath.c_str(), &oStatBuf);

   if( result != 0 )
   {
      ETG_TRACE_ERR(("checkFilePermission has failed with error = %d", result));
   }
   else
   {
      // Print out current "owner", "group" and "permissions" of database file
      ETG_TRACE_USR1(( "checkFilePermission: current 'owner'= '%d' ", oStatBuf.st_uid ));
      ETG_TRACE_USR1(( "checkFilePermission: current 'group'= '%d' ", oStatBuf.st_gid ));
      ETG_TRACE_USR1(( "checkFilePermission: current 'permissions'= '%d' (octal)",
            ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
   }

   return result;
}

int checkAndAdaptFilePermission(const std::string& filePath, const std::string& userName,
      const std::string& groupName, const unsigned int filePermissions)
{
   ETG_TRACE_USR1(("checkAndAdaptFilePermission() FilePath : %s", filePath.c_str()));
   ETG_TRACE_USR1(("checkAndAdaptFilePermission() UserName : %25s, GroupName : %25s",
         userName.c_str(), groupName.c_str()));
   ETG_TRACE_USR1(("checkAndAdaptFilePermission: FilePermissions : %d (octal)",
         ((filePermissions>>6)&0x07)*100 + ((filePermissions>>3)&0x07)*10 + (filePermissions&0x07)));

   // BEGIN: Check and (if necessary) adapt the given file to the wanted "permissions", "owner" and
   // "group" settings (multi-user target)
   int result = 0;

   if((true == filePath.empty()) || (true == userName.empty()) || (true == groupName.empty()) || (0 == filePermissions))
   {
      ETG_TRACE_ERR(( "checkAndAdaptFilePermission has invalid parameters"));
      return -1;
   }

   struct stat oStatBuf;

   // Retrieve current "owner", "group" and "permissions" of the given file
   result = stat(filePath.c_str(), &oStatBuf);

   if( result != 0 )
   {
      ETG_TRACE_ERR(( "checkAndAdaptFilePermission has failed with error = %d", result));
   }
   else
   {
      // Print out current "owner", "group" and "permissions" of database file
      ETG_TRACE_USR1(("checkAndAdaptFilePermission: current 'owner'= '%d' ", oStatBuf.st_uid ));
      ETG_TRACE_USR1(("checkAndAdaptFilePermission: current 'group'= '%d' ", oStatBuf.st_gid ));
      ETG_TRACE_USR1(("checkAndAdaptFilePermission: current permissions= '%d' (octal)",
            ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07)));

      // Check if current "permissions" of the given file matches to the wanted ones,
      // if not try to adapt accordingly
      if((oStatBuf.st_mode & 0x1FF) != (filePermissions))
      {
         // Set the given access for the given file
         ETG_TRACE_ERR(( "checkAndAdaptFilePermission: correcting the file 'permissions' to %d ",
               ((filePermissions>>6)&0x07)*100 + ((filePermissions>>3)&0x07)*10 + (filePermissions&0x07)));

         result = chmod(filePath.c_str(), filePermissions);

         if( result != 0 )
         {
            ETG_TRACE_ERR(( "checkAndAdaptFilePermission: 'chmod' has failed with error = %d", result));
         }
         else
         {
            ETG_TRACE_USR1(("checkAndAdaptFilePermission: 'chmod' successful"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("checkAndAdaptFilePermission: file 'permissions' are correct"));
      }

      struct passwd *poWantedPasswd = nullptr;
      struct group  *poWantedGroup  = nullptr;

      // Check if current "owner" and "group" matches to the wanted ones, if not try to adapt accordingly
      poWantedPasswd = getpwnam(userName.c_str());
      poWantedGroup  = getgrnam(groupName.c_str());

      if(poWantedPasswd && poWantedGroup)
      {
         // Check if current "owner" and "group" for database are correct. If not, then do the needed adaptations.
         if( (poWantedPasswd->pw_uid != oStatBuf.st_uid) || (poWantedGroup->gr_gid != oStatBuf.st_gid) )
         {
            // Change "owner" and "group" settings if current "owner/group" settings are not correct.
            ETG_TRACE_ERR(( "checkAndAdaptFilePermission: correcting the 'owner' and 'group' settings of file"));

            result = chown(filePath.c_str(), poWantedPasswd->pw_uid, poWantedGroup->gr_gid);

            if( result != 0 )
            {
               ETG_TRACE_ERR(( "checkAndAdaptFilePermission: 'chown' has failed with error = %d", result));
            }
            else
            {
               ETG_TRACE_USR1(("checkAndAdaptFilePermission: 'chown' successful"));
            }
         }
         else
         {
            ETG_TRACE_USR1(("checkAndAdaptFilePermission: file 'owner' and 'group' settings are correct"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("checkAndAdaptFilePermission: cannot resolve the 'wanted' user-name = '%s'", userName.c_str()));
         ETG_TRACE_ERR(("checkAndAdaptFilePermission:   or the 'wanted' group-name = '%s'", groupName.c_str() ));
      }
   }

   // Print out "owner", "group" and "permissions" of the given file (after security check) and
   // related changes might have been done.
   checkFilePermission(filePath);

   // END: Check and (if necessary) adapt the given file to the wanted "permissions",
   // "owner" and "group" settings (multi-user target)

   return result;
}

bool getFileNameFromFilePath(const std::string& filePath, std::string& fileName)
{
   ETG_TRACE_USR1(("getFileNameFromFilePath() FilePath : %s", filePath.c_str()));

   bool fileNameFound = false;

   if(true != filePath.empty())
   {
      fileName = filePath;

      // Remove directory if present.
      const size_t dirEndPos = fileName.find_last_of("/");
      if (std::string::npos != dirEndPos)
      {
         fileName.erase(0, dirEndPos + 1);
      }

      // Remove extension if present.
      const size_t extensionPos = fileName.rfind('.');
      if (std::string::npos != extensionPos)
      {
         fileName.erase(extensionPos);
      }

      fileNameFound = true;
   }

   ETG_TRACE_USR1(("getFileNameFromFilePath() FileName : %s", fileName.c_str()));

   return fileNameFound;
}

} // namespace pmcommon
} // namespace bosch
} // namespace com
