/**
 * @file TimerData.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the TimerData class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PM Application- PM Common
 */

#ifndef TimerData_h
#define TimerData_h

#define PERIODIC_TIMER 0x00
#define DEFAULT_TIMER_DURATION 0x00

namespace com
{
namespace bosch
{
namespace pmcommon
{

template <typename TTimerInfo, typename TDuration, typename TRepeatCount>
class TimerData
{
public:
   /**
    * Constructor of TimerData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   TimerData() : _timerInfo(), _duration(DEFAULT_TIMER_DURATION), _repeatCount(PERIODIC_TIMER)
   {

   }

   /**
    * Parameterized Constructor of TimerData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   TimerData(const TTimerInfo timerId, const TDuration duration, const TRepeatCount repeatCount) :
      _timerInfo(timerId), _duration(duration), _repeatCount(repeatCount)
   {

   }

   /**
    * Copy Constructor of TimerData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   TimerData(const TimerData<TTimerInfo, TDuration, TRepeatCount>& other) :
      _timerInfo(other._timerInfo), _duration(other._duration), _repeatCount(other._repeatCount)
   {

   }

   /**
    * Destructor of TimerData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~TimerData()
   {

   }

   /**
    * Friend function used when copying the object.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   friend void swap(TimerData<TTimerInfo, TDuration, TRepeatCount>& first,
         TimerData<TTimerInfo, TDuration, TRepeatCount>& second)
   {
      using std::swap;

      swap(first._timerInfo, second._timerInfo);
      swap(first._duration, second._duration);
      swap(first._repeatCount, second._repeatCount);
   }

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   TimerData<TTimerInfo, TDuration, TRepeatCount>& operator=(TimerData<TTimerInfo, TDuration, TRepeatCount> other)
   {
      swap(*this, other);

      return *this;
   }

   /**
    * Overloading '==' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator==(const TimerData<TTimerInfo, TDuration, TRepeatCount>& rhs) const
   {
      return ((this->_timerInfo == rhs._timerInfo)
            && (this->_duration == rhs._duration))
            && (this->_repeatCount == rhs._repeatCount);
   }

   /**
    * Overloading '!=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator!=(const TimerData<TTimerInfo, TDuration, TRepeatCount>& rhs) const
   {
      return (!(operator==(rhs)));
   }

   // When multiple timers are used in a single class, this timer info will be helpful in
   // distinguishing the different timers during the timer expiry event.
   TTimerInfo _timerInfo;

   // Time duration in milliseconds
   TDuration _duration;

   // Number of times, the timer should expire
   TRepeatCount _repeatCount;
};

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif // TimerData_h
