/**
 * @file LoopbackData.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the prototype of the LoopbackData class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details Any class that would need to post a loopback message to the PM service
 *          shall inherit this class
 *
 * @ingroup PM Application- PM Common
 */

#ifndef LoopbackData_h
#define LoopbackData_h

typedef unsigned char LoopbackMessageId;

namespace com
{
namespace bosch
{
namespace pmcommon
{

/*
 * PmSubComponentId to determine the PM sub component on receiving the loopback message from ASF framework
 */
typedef enum{
   Unknown = 0,
   IpcWrapper,
   PmCore,
   PmAudioManager
}PmSubComponentId;

class LoopbackData
{
public:
   /**
    * Constructor of LoopbackData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   LoopbackData() : _pmSubComponentId(Unknown), _loopbackMsgId(0x00)
   {
   }

   /**
    * Parameterized Constructor of LoopbackData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   LoopbackData(const PmSubComponentId componentId, const LoopbackMessageId msgId) :
      _pmSubComponentId(componentId), _loopbackMsgId(msgId)
   {
   }

   /**
    * Copy Constructor of LoopbackData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   LoopbackData(const LoopbackData& other) : _pmSubComponentId(other._pmSubComponentId),
         _loopbackMsgId(other._loopbackMsgId)
   {
   }

   /**
    * Destructor of LoopbackData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~LoopbackData()
   {
   }

   /**
    * Friend function used when copying the object.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   friend void swap(LoopbackData& first, LoopbackData& second)
   {
      using std::swap;

      swap(first._pmSubComponentId, second._pmSubComponentId);
      swap(first._loopbackMsgId, second._loopbackMsgId);
   }

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   LoopbackData& operator=(LoopbackData other)
   {
      swap(*this, other);

      return *this;
   }

   /**
    * Overloading '==' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   bool operator==(const LoopbackData& rhs) const
   {
      return ((this->_pmSubComponentId == rhs._pmSubComponentId)
            &&(this->_loopbackMsgId == rhs._loopbackMsgId));
   }

   inline PmSubComponentId getPmSubComponent(void) const
   {
      return _pmSubComponentId;
   }

   inline void setPmSubComponent(const PmSubComponentId subComponentId)
   {
      _pmSubComponentId = subComponentId;
   }

   inline LoopbackMessageId getLoopbackMessageId(void) const
   {
      return _loopbackMsgId;
   }

   inline void setLoopbackMessageId(const LoopbackMessageId messageId)
   {
      _loopbackMsgId = messageId;
   }

   virtual void traceMessage(void) const
   {
   }

protected:
   PmSubComponentId _pmSubComponentId; /**< Component ID to distinguish the PM sub component */
   LoopbackMessageId _loopbackMsgId;   /**< ID of the loopback message */
};

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif // LoopbackData_h
