/**
 * @file PhoneCallManager.cpp
 *
 * @swcomponent PhoneCallManager 
 *
 * @brief This file contains the definition of the PhoneCallManager class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details The instances of all IpcWrapper layer related classes of PM Application are created here
 *
 * @ingroup Phone Module
 */

#include "PhoneCallManager.h"
#include "PmCoreRequestIf.h"
#include "DeviceDetailsListHandler.h"
#include "PmConfiguration.h"
#include "IpcWrapper.h"
#include "JsonReader.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_PM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PhoneCallManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_PM_APPLICATION
#endif
#endif

#define PHONE_CALL_MANAGER_REGISTRY_NAME "/dev/root/opt/bosch/base/registry/fc_phone.reg"

const ::std::string PmConfigFile = "/var/opt/bosch/static/connectivity/config/PM_config.json";

PhoneCallManager::PhoneCallManager()
{
   ETG_TRACE_USR4(("PhoneCallManager::PhoneCallManager() entered"));

   amt_bInit();

   scd_init();

   buildRegistry();

   //Create object for PmConfiguration and DataProvider Classes
   com::bosch::pmcommon::PmConfiguration::getInstance();
   com::bosch::pmcommon::JsonReader::getInstance().readConfiguration(PmConfigFile);
   com::bosch::pmcommon::JsonReader::getInstance().traceConfiguration();

   _pPmAppCcaServer = new PmAppCcaServer;

   _pPmAppAsfDbusServer = new PmAppAsfDbusServer;

   _pIpcMessageDispatcher = new IpcMessageDispatcher;
   _pIpcMessageDispatcher->registerMsgHandler(_pPmAppCcaServer->getCcaMsgHandler());

   com::bosch::pmapp::DeviceDetailsListHandler::getInstance();
   pm_ipc_wrapper::IpcWrapper::getInstance().registerIpcMessageDispatcher(_pIpcMessageDispatcher);
   pm_ipc_wrapper::IpcWrapper::getInstance().setPmApplication(static_cast<ahl_tclBaseOneThreadApp*>(
         _pPmAppCcaServer->getPmApplication()));

   //ClientHandlers should be invoked at last after initializing IpcWrapper and PmCore
   _pPmAppClientHandler = new PmAppClientHandler;
}

IpcMessageDispatcher* PhoneCallManager::getIpcMessageDispatcher()
{
   ETG_TRACE_USR4(("PhoneCallManager::getIpcMessageDispatcher() entered"));
   return _pIpcMessageDispatcher;
}

void PhoneCallManager::buildRegistry() const
{
   ETG_TRACE_USR4(("buildRegistry() entered"));

   OSAL_tIODescriptor regHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_REGISTRY, OSAL_EN_READWRITE);

   if(regHandle != -1) //FIXME:OSAL-Generic Macro has to defined later for -1
   {
      //FIXME:OSAL-Generic Macro has to defined later for 0
      if((OSAL_s32IOControl(regHandle, OSAL_C_S32_IOCTRL_BUILD_REG, (intptr_t) PHONE_CALL_MANAGER_REGISTRY_NAME)) == 0)
      {
         ETG_TRACE_USR4(("REGISTRY loaded '%s'", PHONE_CALL_MANAGER_REGISTRY_NAME));
      }
      else
      {
         ETG_TRACE_ERR(("FAILED to load REGISTRY '%s'", PHONE_CALL_MANAGER_REGISTRY_NAME));
      }

      OSAL_s32IOClose(regHandle);
   }
   else
   {
      ETG_TRACE_ERR(("FAILED to access the device registry "));
   }
}

PhoneCallManager::~PhoneCallManager()
{
   ETG_TRACE_USR4(("PhoneCallManager::~PhoneCallManager() entered"));

   if(_pPmAppCcaServer)
   {
      delete _pPmAppCcaServer;
      _pPmAppCcaServer = nullptr;
   }

   if(_pIpcMessageDispatcher)
   {
      delete _pIpcMessageDispatcher;
      _pIpcMessageDispatcher = nullptr;
   }

   if(_pPmAppClientHandler)
   {
      delete _pPmAppClientHandler;
      _pPmAppClientHandler = nullptr;
   }

   if(_pPmAppAsfDbusServer)
   {
      delete _pPmAppAsfDbusServer;
      _pPmAppAsfDbusServer = nullptr;
   }
}
