/**
 * @file WaitingModeToneAudioHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the WaitingModeToneAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of waiting mode tone audio
 *
 * @ingroup PmAudioManager
 */

#include "WaitingModeToneAudioHandler.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "ArlWrapper.h"
#include "AudioPlayerWrapper.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WaitingModeToneAudioHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   WaitingModeToneAudioHandler::WaitingModeToneAudioHandler()
   {
      _smAudioWrapperReqIf = nullptr;
      _smAudioPlayerReqIf = nullptr;
      _smWaitingModeToneSmHandlerIf = nullptr;
      _audioSessionHandler = nullptr;
      _smTraceObserverIf = nullptr;
   }

   WaitingModeToneAudioHandler::WaitingModeToneAudioHandler(AudioSessionHandler* audioSessionHandler)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: Constructor"));

      _smAudioWrapperReqIf = new WaitingModeToneAudioHandler::SmAudioWrapperReqIf();

      if(nullptr != _smAudioWrapperReqIf)
      {
         setSCI_AudioWrapperReqIf_OCB(_smAudioWrapperReqIf);
      }

      _smAudioPlayerReqIf = new WaitingModeToneAudioHandler::SmAudioPlayerReqIf(*this);

      if(nullptr != _smAudioPlayerReqIf)
      {
         setSCI_AudioPlayerReqIf_OCB(_smAudioPlayerReqIf);
      }

      _smWaitingModeToneSmHandlerIf = new WaitingModeToneAudioHandler::SmWaitingModeToneSmHandlerIf(*this);

      if(nullptr != _smWaitingModeToneSmHandlerIf)
      {
         setSCI_WaitingModeToneSmHandler_OCB(_smWaitingModeToneSmHandlerIf);
      }

      _audioSessionHandler = audioSessionHandler;

      _smTraceObserverIf = new WaitingModeToneAudioHandler::SmTraceObserverIf<WaitingModeToneAudioSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   WaitingModeToneAudioHandler::~WaitingModeToneAudioHandler()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: Destructor"));

      if(nullptr != _smAudioWrapperReqIf)
      {
         delete _smAudioWrapperReqIf;
         _smAudioWrapperReqIf = nullptr;
      }

      if(nullptr != _smAudioPlayerReqIf)
      {
         delete _smAudioPlayerReqIf;
         _smAudioPlayerReqIf = nullptr;
      }

      if(nullptr != _smWaitingModeToneSmHandlerIf)
      {
         delete _smWaitingModeToneSmHandlerIf;
         _smWaitingModeToneSmHandlerIf = nullptr;
      }

      _audioSessionHandler = nullptr;

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void WaitingModeToneAudioHandler::prepareWaitingModeToneAudio()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: prepareWaitingModeToneAudio"));

      getSCI_AmClientRequest()->raise_pREPARE_WAITING_MODE_TONE_CHANNEL();
   }

   void WaitingModeToneAudioHandler::pauseWaitingModeToneAudio()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: pauseWaitingModeToneAudio"));

      getSCI_AmClientRequest()->raise_pAUSE_WAITING_MODE_TONE();
   }

   void WaitingModeToneAudioHandler::stopWaitingModeToneAudio()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: stopWaitingModeToneAudio"));

      getSCI_AmClientRequest()->raise_sTOP_WAITING_MODE_TONE();
   }

   void WaitingModeToneAudioHandler::playAudioFile(const FilePath& filePath)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: playAudioFile"));

      getSCI_AmClientRequest()->raise_pLAY_WAITING_MODE_TONE(filePath);
   }

   void WaitingModeToneAudioHandler::onSourceActivityResult(const SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: onSourceActivityResult"));

      switch(srcActivity)
      {
         case SRC_ACTIVITY_OFF:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_SA_OFF();
            break;
         case SRC_ACTIVITY_ON:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_SA_ON();
            break;
         case SRC_ACTIVITY_PAUSE:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_SA_PAUSE();
            break;
         default:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_SA_OFF();
            break;
      }
   }

   void WaitingModeToneAudioHandler::onAudioManagerResponse(const AudioChannelStatus status,
         const AudioSink& audioSink)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:onAudioManagerResponse : %u",
            ETG_CENUM(AudioChannelStatus, status)));

      switch(status)
      {
         case CHANNEL_GRANTED:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_CHANNEL_GRANTED(audioSink);
            break;
         case CHANNEL_DENIED:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_CHANNEL_DENIED();
            break;
         case CHANNEL_DEALLOCATED: //intentional fall through
         default:
            getSCI_AudioWrapperResponse()->raise_wAITING_MODE_TONE_CHANNEL_DEALLOCATED();
            break;
      }
   }

   void WaitingModeToneAudioHandler::onAudioPlayerResponse(const PlayStatus playStatus,
         const AudioPlayerErrorInfo& errorInfo)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler::onAudioPlayerResponse : PlayStatus: %u, errorInfo: %50s",
            ETG_CENUM(PlayStatus, playStatus), errorInfo.c_str()));

      switch(playStatus)
      {
         case PlayStatus::PLAYING:
            getSCI_AudioPlayerResponse()->raise_pLAYING_SUCCESS();
            break;
         case PlayStatus::STOPPED:
            getSCI_AudioPlayerResponse()->raise_sTOP_SUCCESS();
            break;
         case PlayStatus::PLAYBACK_ERROR:
            getSCI_AudioPlayerResponse()->raise_pLAYING_ERROR(errorInfo);  // Fixme: The type in SM defined event - PLAYING_ERROR
            break;
         case PlayStatus::STOP_PLAYBACK_ERROR:
            getSCI_AudioPlayerResponse()->raise_sTOP_FAILURE(errorInfo);   // Fixme: The type in SM defined event - STOP_FAILURE
            break;
         default:
            break;
      }
   }

   void WaitingModeToneAudioHandler::SmAudioWrapperReqIf::sendAllocateRequest()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: sendAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().requestAudioRoute(AM_WAITINGMODE);
   }

   void WaitingModeToneAudioHandler::SmAudioWrapperReqIf::sendDeAllocateRequest()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: sendDeAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().releaseAudioRoute(AM_WAITINGMODE);
   }

   void WaitingModeToneAudioHandler::SmAudioWrapperReqIf::sendSrcActivityResult(SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler: sendSrcActivityResult"));

      (void)AmMainController::getInstance().getArlWrapper().sendSourceActivityResult(AM_WAITINGMODE, srcActivity);
   }

   AmResultCode WaitingModeToneAudioHandler::SmAudioPlayerReqIf::setAlsaDeviceName(AudioSink sinkName,
         AudioSource srcName)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:setAlsaDeviceName: Sink : %50s, Source : %50s",
            sinkName.c_str(), srcName.c_str()));

      return AmMainController::getInstance().getAudioPlayerWrapper().setAlsaDeviceRequest(
            _waitingModeToneAudioHandler._audioSessionHandler->getAmSessionId(), sinkName, srcName, WAITINGMODE_TONE);
   }

   AmResultCode WaitingModeToneAudioHandler::SmAudioPlayerReqIf::sendPlayFileRequest(FilePath filePath)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:sendPlayFileRequest : \"%80s\"", filePath.c_str()));

      return AmMainController::getInstance().getAudioPlayerWrapper().playAudioFileRequest(
            _waitingModeToneAudioHandler._audioSessionHandler->getAmSessionId(), filePath, PLAY_COUNT_DEFAULT);
   }

   AmResultCode WaitingModeToneAudioHandler::SmAudioPlayerReqIf::stopPlayback()
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:stopPlayback"));

      return AmMainController::getInstance().getAudioPlayerWrapper().stopPlaybackRequest(
            _waitingModeToneAudioHandler._audioSessionHandler->getAmSessionId());
   }

   void WaitingModeToneAudioHandler::SmWaitingModeToneSmHandlerIf::updateWaitingModeToneFilePath(FilePath filePath)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:updateWaitingModeToneFilePath : \"%80s\"", filePath.c_str()));

      _waitingModeToneAudioHandler.getSCI_WaitingModeToneSmHandler()->set_waitingModeToneFilePath(filePath);
   }

   void WaitingModeToneAudioHandler::SmWaitingModeToneSmHandlerIf::updateStatus(RouteState routeState,
         AmResultMessage msg)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:updateStatus : RouteState : %u, Message : %50s",
            ETG_CENUM(RouteState, routeState), msg.c_str()));

      if(nullptr != _waitingModeToneAudioHandler._audioSessionHandler)
      {
         _waitingModeToneAudioHandler._audioSessionHandler->updateStatus(AM_WAITINGMODE, routeState, msg);
      }
   }

   void WaitingModeToneAudioHandler::SmWaitingModeToneSmHandlerIf::updateSinkName(AudioSink sinkName)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler:updateSinkName : \"%80s\"", sinkName.c_str()));

      _waitingModeToneAudioHandler.getSCI_WaitingModeToneSmHandler()->set_sinkName(sinkName);
   }

   template<typename T>
   void WaitingModeToneAudioHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler::stateEntered()::%u",
            ETG_CENUM(WaitingModeToneAudioSmStates, state)));
   }

   template<typename T>
   void WaitingModeToneAudioHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("WaitingModeToneAudioHandler::stateExited()::%u",
            ETG_CENUM(WaitingModeToneAudioSmStates, state)));
   }
}
