/**
 * @file VoiceCallAudioHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VoiceCallAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call audio
 *
 * @ingroup PmAudioManager
 */

#include "VoiceCallAudioHandler.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "ArlWrapper.h"
#include "EcnrWrapper.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/VoiceCallAudioHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   VoiceCallAudioHandler::VoiceCallAudioHandler()
   {
      _smAudioWrapperReqIf = nullptr;
      _smEcnrRequestIf = nullptr;
      _smVoiceCallAudioSmHandlerIf = nullptr;
      _audioSessionHandler = nullptr;
      _smTraceObserverIf = nullptr;
   }

   VoiceCallAudioHandler::VoiceCallAudioHandler(AudioSessionHandler* audioSessionHandler)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler: Constructor"));

      _smAudioWrapperReqIf = new VoiceCallAudioHandler::SmAudioWrapperReqIf();

      if(nullptr != _smAudioWrapperReqIf)
      {
         setSCI_AudioWrapperReqIf_OCB(_smAudioWrapperReqIf);
      }

      _smEcnrRequestIf = new VoiceCallAudioHandler::SmEcnrRequestIf(*this);

      if(nullptr != _smEcnrRequestIf)
      {
         setSCI_EcnrRequest_OCB(_smEcnrRequestIf);
      }

      _smVoiceCallAudioSmHandlerIf = new VoiceCallAudioHandler::SmVoiceCallAudioSmHandlerIf(*this);

      if(nullptr != _smVoiceCallAudioSmHandlerIf)
      {
         setSCI_VoiceCallAudioSmHandler_OCB(_smVoiceCallAudioSmHandlerIf);
      }

      _audioSessionHandler = audioSessionHandler;

      _smTraceObserverIf = new VoiceCallAudioHandler::SmTraceObserverIf<VoiceCallAudioSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   VoiceCallAudioHandler::~VoiceCallAudioHandler()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler: Destructor"));

      if(nullptr != _smAudioWrapperReqIf)
      {
         delete _smAudioWrapperReqIf;
         _smAudioWrapperReqIf = nullptr;
      }

      if(nullptr != _smEcnrRequestIf)
      {
         delete _smEcnrRequestIf;
         _smEcnrRequestIf = nullptr;
      }

      if(nullptr != _smVoiceCallAudioSmHandlerIf)
      {
         delete _smVoiceCallAudioSmHandlerIf;
         _smVoiceCallAudioSmHandlerIf = nullptr;
      }

      _audioSessionHandler = nullptr;

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void VoiceCallAudioHandler::prepareVoiceCallAudio()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:prepareVoiceCallAudio"));

      getSCI_AmClientRequest()->raise_pREPARE_VOICECALL_CHANNEL();
   }

   void VoiceCallAudioHandler::pauseVoiceCallAudio()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:pauseVoiceCallAudio"));

      getSCI_AmClientRequest()->raise_pAUSE_VOICECALL_AUDIO();
   }

   void VoiceCallAudioHandler::stopVoiceCallAudio()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:stopVoiceCallAudio"));

      getSCI_AmClientRequest()->raise_sTOP_VOICECALL_AUDIO();
   }

   void VoiceCallAudioHandler::playHfAudio(const SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:playHfAudio : %u",
            ETG_CENUM(SamplingType, samplingType)));

      getSCI_AmClientRequest()->raise_sTREAM_VOICECALL_AUDIO(samplingType);
   }

   void VoiceCallAudioHandler::onAudioManagerResponse(const AudioChannelStatus status)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:onAudioManagerResponse: %u",
            ETG_CENUM(AudioChannelStatus, status)));

      switch(status)
      {
         case CHANNEL_GRANTED:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_CHANNEL_GRANTED();
            break;
         case CHANNEL_DENIED:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_CHANNEL_DENIED();
            break;
         case CHANNEL_DEALLOCATED: //intentional fall through
         default:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_CHANNEL_DEALLOCATED();
            break;
      }
   }

   void VoiceCallAudioHandler::onSourceActivityResult(const SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:onSourceActivityResult: SrcActivity : %u",
            ETG_CENUM(SrcActivity, srcActivity)));

      switch(srcActivity)
      {
         case SRC_ACTIVITY_OFF:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_SA_OFF();
            break;
         case SRC_ACTIVITY_ON:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_SA_ON();
            break;
         case SRC_ACTIVITY_PAUSE:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_SA_PAUSE();
            break;
         default:
            getSCI_AudioWrapperResponse()->raise_vOICECALL_AUDIO_SA_OFF();
            break;
      }
   }

   void VoiceCallAudioHandler::onEcnrResponse(const EcnrReponse response, const AmResultMessage& msg)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:onEcnrResponse: %u", ETG_CENUM(EcnrReponse, response)));

      switch(response)
      {
         case ECNR_INIT_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_INIT_FAILURE(msg);
            break;
         case ECNR_INIT_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_INIT_SUCCESS();
            break;
         case ECNR_DESTROY_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_DESTROY_FAILURE(msg);
            break;
         case ECNR_DESTROY_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_DESTROY_SUCCESS();
            break;
         case ECNR_SET_CONFIGURATION_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_SET_CONFIGURATION_FAILURE(msg);
            break;
         case ECNR_SET_CONFIGURATION_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_SET_CONFIGURATION_SUCCESS();
            break;
         case ECNR_START_AUDIO_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_START_FAILURE(msg);
            break;
         case ECNR_START_AUDIO_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_START_SUCCESS();
            break;
         case ECNR_STOP_AUDIO_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_STOP_FAILURE(msg);
            break;
         case ECNR_STOP_AUDIO_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_STOP_SUCCESS();
            break;
         default:
            break;
      }
   }

   void VoiceCallAudioHandler::SmAudioWrapperReqIf::sendAllocateRequest()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().requestAudioRoute(AM_PHONEAUDIO);
   }

   void VoiceCallAudioHandler::SmAudioWrapperReqIf::sendDeAllocateRequest()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendDeAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().releaseAudioRoute(AM_PHONEAUDIO);
   }

   void VoiceCallAudioHandler::SmAudioWrapperReqIf::sendSrcActivityResult(SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendSrcActivityResult: SrcActivity : %u",
            ETG_CENUM(SrcActivity, srcActivity)));

      (void)AmMainController::getInstance().getArlWrapper().sendSourceActivityResult(AM_PHONEAUDIO, srcActivity);
   }

   AmResultCode VoiceCallAudioHandler::SmEcnrRequestIf::sendEcnrInit(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendEcnrInit: SamplingType : %u",
            ETG_CENUM(SamplingType, samplingType)));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         auto& pmConfiguration = com::bosch::pmcommon::PmConfiguration::getInstance();
         EcnrConfigurationId configId = pmConfiguration.getEcnrDataSetForWBSPhoneCall();

         if(NARROWBAND == samplingType)
         {
            configId = pmConfiguration.getEcnrDataSetForNBSPhoneCall();
         }

         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().initialize(
               _voiceCallAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, configId))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceCallAudioHandler::SmEcnrRequestIf::sendEcnrSetConfiguration(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendEcnrSetConfiguration:  SamplingType : %u",
            ETG_CENUM(SamplingType, samplingType)));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         auto& pmConfiguration = com::bosch::pmcommon::PmConfiguration::getInstance();
         EcnrConfigurationId configId = pmConfiguration.getEcnrDataSetForWBSPhoneCall();

         if(NARROWBAND == samplingType)
         {
            configId = pmConfiguration.getEcnrDataSetForNBSPhoneCall();
         }

         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().setConfiguration(
               _voiceCallAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, configId))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceCallAudioHandler::SmEcnrRequestIf::sendEcnrStart()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendEcnrStart"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().startAudio(
               _voiceCallAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceCallAudioHandler::SmEcnrRequestIf::sendEcnrStop()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendEcnrStop"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().stopAudio(
               _voiceCallAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceCallAudioHandler::SmEcnrRequestIf::sendEcnrDestroy()
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:sendEcnrDestroy"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().destroy(
               _voiceCallAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   void VoiceCallAudioHandler::SmVoiceCallAudioSmHandlerIf::updateSamplingType(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:updateSamplingType : %u", ETG_CENUM(SamplingType, samplingType)));

      _voiceCallAudioHandler.getSCI_VoiceCallAudioSmHandler()->set_samplingType(samplingType);
   }

   void VoiceCallAudioHandler::SmVoiceCallAudioSmHandlerIf::updateStatus(RouteState routeState,
         AmResultMessage msg)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler:updateStatus : RouteState : %u, Message : %50s",
            ETG_CENUM(RouteState, routeState), msg.c_str()));

      if(nullptr != _voiceCallAudioHandler._audioSessionHandler)
      {
         _voiceCallAudioHandler._audioSessionHandler->updateStatus(AM_PHONEAUDIO, routeState, msg);
      }
   }

   template<typename T>
   void VoiceCallAudioHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler::stateEntered()::%u", ETG_CENUM(VoiceCallAudioSmStates, state)));
   }

   template<typename T>
   void VoiceCallAudioHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("VoiceCallAudioHandler::stateExited()::%u", ETG_CENUM(VoiceCallAudioSmStates, state)));
   }
}
