/**
 * @file WaitingModeToneAudioHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the WaitingModeToneAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of waiting mode tone audio
 *
 * @ingroup PmAudioManager
 */

#ifndef WaitingModeToneAudioHandler_h
#define WaitingModeToneAudioHandler_h

#include "AmIfMessageResult.h"
#include "../src-gen/WaitingModeToneAudioSm.h"

namespace pmaudiomanager
{
   //class forward declaration
   class AudioSessionHandler;

   class WaitingModeToneAudioHandler final: public WaitingModeToneAudioSm
   {
   public:
      /**
       * Parameterized Constructor of WaitingModeToneAudioHandler class
       *
       * @param[in] audioSessionHandler - pointer of AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      WaitingModeToneAudioHandler(AudioSessionHandler* audioSessionHandler);

      /**
       * Destructor of WaitingModeToneAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~WaitingModeToneAudioHandler();

      /**
       * This method is used to notify the SM to prepare WaitingModeTone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareWaitingModeToneAudio();

      /**
       * This method is used to notify the SM to pause WaitingModeTone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseWaitingModeToneAudio();

      /**
       * This method is used to notify the SM to stop WaitingModeTone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopWaitingModeToneAudio();

      /**
       * This method is used to notify the SM to play given audio file
       *
       * @param[in] filePath - file path
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playAudioFile(const FilePath& filePath);

      /**
       * This method is used to notify the audio source activity changes to SM
       *
       * @param[in] srcActivity - source activity
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSourceActivityResult(const SrcActivity srcActivity);

      /**
       * This method is used to notify the audio channel allocation/deallocation status to SM
       *
       * @param[in] status - audio channel allocation/deallocation status
       * @param[in] audioSink - audio sink device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerResponse(const AudioChannelStatus status, const AudioSink& audioSink = "");

      /**
       * This method is used to notify the AudioPlayer response to SM
       *
       * @param[in] playStatus - playback status
       * @param[in] errorInfo  - error info (default - empty string)
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioPlayerResponse(const PlayStatus playStatus, const AudioPlayerErrorInfo& errorInfo = "");

   private:
      /**
       * Constructor of WaitingModeToneAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      WaitingModeToneAudioHandler();

      /**
       * SmAudioWrapperReqIf class inherits the functions of SCI_AudioWrapperReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioWrapperReqIf final: public SCI_AudioWrapperReqIf_OCB
      {
      public:
         SmAudioWrapperReqIf(){}

         ~SmAudioWrapperReqIf(){}

         virtual void sendAllocateRequest() override;

         virtual void sendDeAllocateRequest() override;

         virtual void sendSrcActivityResult(SrcActivity srcActivity) override;
      };

      /**
       * SmAudioPlayerReqIf class inherits the functions of SCI_AudioPlayerReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioPlayerReqIf final: public SCI_AudioPlayerReqIf_OCB
      {
      public:
         SmAudioPlayerReqIf(WaitingModeToneAudioHandler& waitingModeToneAudioHandler) :
            _waitingModeToneAudioHandler(waitingModeToneAudioHandler){}

         ~SmAudioPlayerReqIf(){}

         virtual AmResultCode setAlsaDeviceName(AudioSink sinkName, AudioSource srcName) override;

         virtual AmResultCode sendPlayFileRequest(FilePath filePath) override;

         virtual AmResultCode stopPlayback() override;

      private:
         WaitingModeToneAudioHandler& _waitingModeToneAudioHandler;
      };

      /**
       * SmWaitingModeToneSmHandlerIf class inherits the functions of SCI_WaitingModeToneSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmWaitingModeToneSmHandlerIf final: public SCI_WaitingModeToneSmHandler_OCB
      {
      public:
         SmWaitingModeToneSmHandlerIf(WaitingModeToneAudioHandler& waitingModeToneAudioHandler):
            _waitingModeToneAudioHandler(waitingModeToneAudioHandler){}

         ~SmWaitingModeToneSmHandlerIf(){}

         virtual void updateWaitingModeToneFilePath(FilePath filePath) override;

         virtual void updateStatus(RouteState routeState, AmResultMessage msg) override;

         virtual void updateSinkName(AudioSink sinkName) override;

      private:
         WaitingModeToneAudioHandler& _waitingModeToneAudioHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final: public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(WaitingModeToneAudioHandler& waitingModeToneAudioHandler):
            _waitingModeToneAudioHandler(waitingModeToneAudioHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         WaitingModeToneAudioHandler& _waitingModeToneAudioHandler;
      };

      AudioSessionHandler*          _audioSessionHandler;          /**< pointer to AudioSessionHandler */
      SmAudioWrapperReqIf*          _smAudioWrapperReqIf;          /**< pointer to SmAudioWrapperReqIf */
      SmAudioPlayerReqIf*           _smAudioPlayerReqIf;           /**< pointer to SmAudioPlayerReqIf */
      SmWaitingModeToneSmHandlerIf* _smWaitingModeToneSmHandlerIf; /**< pointer to SmWaitingModeToneSmHandlerIf */
      SmTraceObserverIf<WaitingModeToneAudioSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //WaitingModeToneAudioHandler_h
