/**
 * @file MuteUnmuteHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the MuteUnmuteHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call audio
 *
 * @ingroup PmAudioManager
 */

#ifndef MuteUnmuteHandler_h
#define MuteUnmuteHandler_h

#include "AmIfMessageResult.h"
#include "../src-gen/MuteUnmuteSm.h"

namespace pmaudiomanager
{
   //class forward declaration
   class AudioSessionHandler;

   class MuteUnmuteHandler final: public MuteUnmuteSm
   {
   public:
      /**
       * Parameterized Constructor of MuteUnmuteHandler class
       *
       * @param[in] audioSessionHandler - pointer of AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      MuteUnmuteHandler(IN AudioSessionHandler* audioSessionHandler);

      /**
       * Destructor of MuteUnmuteHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~MuteUnmuteHandler();

      /**
       * This method is used to notify the SM to mute the AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void muteAudio();

      /**
       * This method is used to notify the SM to demute the AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void deMuteAudio();

      /**
       * This method is used to notify the Ecnr response to SM
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onEcnrResponse(const EcnrReponse response, const AmResultMessage& msg);

   private:
      /**
       * Constructor of MuteUnmuteHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      MuteUnmuteHandler();

      /**
       * SmEcnrRequestIf class inherits the functions of SCI_EcnrRequest_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmEcnrRequestIf final: public SCI_EcnrRequest_OCB
      {
      public:
         SmEcnrRequestIf(MuteUnmuteHandler& MuteUnmuteHandler):
            _MuteUnmuteHandler(MuteUnmuteHandler){}

         ~SmEcnrRequestIf(){}

         virtual AmResultCode sendEcnrMuteRequest() override;

         virtual AmResultCode sendEcnrDeMuteRequest() override;

      private:
         MuteUnmuteHandler& _MuteUnmuteHandler;
      };

      /**
       * SmMuteUnmuteSmHandlerIf class inherits the functions of SCI_MuteUnmuteSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class MuteUnmuteSmHandlerIf final: public SCI_SmHandler_OCB
      {
      public:
         MuteUnmuteSmHandlerIf(MuteUnmuteHandler& muteUnmuteHandler):
            _MuteUnmuteHandler(muteUnmuteHandler){}

         ~MuteUnmuteSmHandlerIf(){}

         virtual void updateStatus(AmResultMessage msg) override;

      private:
         MuteUnmuteHandler& _MuteUnmuteHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final : public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(MuteUnmuteHandler& MuteUnmuteHandler):
            _MuteUnmuteHandler(MuteUnmuteHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         MuteUnmuteHandler& _MuteUnmuteHandler;
      };

      AudioSessionHandler*         _audioSessionHandler;         /**< pointer to AudioSessionHandler */
      SmEcnrRequestIf*             _smEcnrRequestIf;             /**< pointer to SmEcnrRequestIf */
      MuteUnmuteSmHandlerIf*       _smMuteUnmuteSmHandlerIf;     /**< pointer to SmMuteUnmuteSmHandlerIf */
      SmTraceObserverIf<MuteUnmuteSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //MuteUnmuteHandler_h
