/**
 * @file PmAudioManagerRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmAudioManagerRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PmAudioManager.
 *
 * @ingroup PmAudioManager
 */


#include "PmAudioManagerRequestIf.h"
#include "AmIfMessageCreator.h"
#include "AmMainController.h"
#include "ahl_fw.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmAudioManagerRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{

PmAudioManagerRequestIf::PmAudioManagerRequestIf()
{
   ETG_TRACE_USR1(("PmAudioManagerRequestIf"));
}

PmAudioManagerRequestIf::~PmAudioManagerRequestIf()
{
   ETG_TRACE_USR1(("~PmAudioManagerRequestIf"));
}

AmResult PmAudioManagerRequestIf::intializePmAudioManager(ahl_tclBaseOneThreadApp* mainApplication)
{
   ETG_TRACE_USR1(("intializePmAudioManager"));
   AmResult amResult(AM_RESULT_OK, "");

   //Initialize AmMainController
   AmMainController::getInstance();
   AmMainController::getInstance().setPmApplication(mainApplication);

   return amResult;
}

AmResult PmAudioManagerRequestIf::registerPmAudioManagerCallbackIf
(IN const IPmAudioManagerCallbackIf* iPmAudioManagerCallbackIf)
{
   ETG_TRACE_USR1(("registerPmAudioManagerCallbackIf"));

   AmResult amResult(AM_RESULT_OK, "");

   amResult = AmMainController::getInstance().registerCallback(iPmAudioManagerCallbackIf);

   return amResult;
}

void PmAudioManagerRequestIf::setAmState(IN const AmState amState)
{
   ETG_TRACE_USR1(("setAmState: amState: %u", amState));
}

AmResult PmAudioManagerRequestIf::prepareAudioRoute(IN const AmSessionId amSessionId,
      IN const AudioChannel audioChannelId)
{
   ETG_TRACE_USR1(("prepareAudioRoute: amSessionId: %u and audioChannelId: %u", amSessionId, audioChannelId));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_PrepareAudioRouteRequest(amSessionId, audioChannelId);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::playAudio(IN const AmSessionId amSessionId, IN const FilePath& filePath,
      IN const PlayCount playCount, IN const ToneType toneType)
{
   ETG_TRACE_USR1(("playAudio: amSessionId: %u and playCount: %u", amSessionId, playCount));
   ETG_TRACE_USR4(("playAudio: filePath: %s", filePath.c_str()));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_PlayAudioFileRequest(amSessionId, filePath, playCount, toneType);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::playAudio(IN const AmSessionId amSessionId, IN const SamplingType samplingType)
{
   ETG_TRACE_USR1(("playAudio: amSessionId: %u and samplingType: %u", amSessionId, samplingType));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_PlayHfAudioRequest(amSessionId, samplingType);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::stopAudio(IN const AmSessionId amSessionId)
{
   ETG_TRACE_USR1(("stopAudio: amSessionId: %u", amSessionId));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_StopAudioRequest(amSessionId);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::setMicMuteState(IN const MuteState muteState)
{
   ETG_TRACE_USR1(("setMicMuteState: muteState: %u", muteState));
   AmResult amResult(AM_RESULT_OK, "");

   /* TODO:
    * To be checked whether setMicMuteState need to be checked in separate State machine.
    * because the Setting the mic to mute is not to be handled for a specific Am Session.
    * It is to be handled generically.
    */

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_SetMicMuteStateRequest(muteState);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::pauseAudio(IN const AmSessionId amSessionId)
{
   ETG_TRACE_USR1(("pauseAudio: amSessionId: %u", amSessionId));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_PauseAudioRequest(amSessionId);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::getAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId)
{
   ETG_TRACE_USR1(("getAudioSinkVolume: amSessionId: %u and sinkId: %u", amSessionId, sinkId));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_GetAudioSinkVolumeRequest(amSessionId, sinkId);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

AmResult PmAudioManagerRequestIf::setAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyVal)
{
   ETG_TRACE_USR1(("setAudioSinkVolume: amSessionId: %u and sinkId: %u", amSessionId, sinkId));
   AmResult amResult(AM_RESULT_OK, "");

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_SetAudioSinkVolumeRequest(amSessionId, sinkId, propertyVal);
   amIfMessage->traceMessage();
   amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   return amResult;
}

void PmAudioManagerRequestIf::onLoopbackMessage(IN const AmActToken /*act*/,
      IN com::bosch::pmcommon::LoopbackData* loopbackData)
{
   ETG_TRACE_USR1(("PmAudioManagerRequestIf::onLoopbackMessage() Entered"));

   //Handle the loopback message
   if(nullptr != loopbackData)
   {
      //Identify the message ID
      switch(loopbackData->getLoopbackMessageId())
      {
         case AM_IF_MSG_ID_PLAYBACK_STATUS:
         {
            //Convert the loopback message into the PlaybackStatus Message
            AmLoopbackMessage_PlaybackStatus* playbackMsg = dynamic_cast<AmLoopbackMessage_PlaybackStatus*>(loopbackData);
            std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_PlaybackStatus(
                  playbackMsg->getPlaybackStatusMsg().getAmSessionId(), playbackMsg->getPlaybackStatusMsg().getAmResult(),
                  playbackMsg->getPlaybackStatusMsg().getPlayStatus(), playbackMsg->getPlaybackStatusMsg().getToneType());
            amIfMessage->traceMessage();

            (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);

            delete playbackMsg;
         }
         break;

         default:
         {
            ETG_TRACE_ERR(("PmAudioManagerRequestIf::onLoopbackMessage : Invalid loopback message"));
            delete loopbackData;
         }
         break;
      }
   }
}

} /* namespace pmaudiomanager */
