/**
 * @file PmAudioManagerTypesInternal.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the internal type definitions of PmAudioManager
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details PmAudioManager type definitions are defined here.
 *
 * @ingroup PmAudioManager
 */

#ifndef PmAudioManagerTypesInternal_h
#define PmAudioManagerTypesInternal_h

#include <string>
#include <map>
#include <memory>
#include <algorithm>

namespace pmaudiomanager
{
/**
 * macros to mark parameters data flow direction
 */
#ifndef IN
#define IN /**< direction marker for input parameter */
#endif
#ifndef OUT
#define OUT /**< direction marker for output parameter */
#endif
#ifndef INOUT
#define INOUT /**< direction marker for input/output parameter */
#endif

#define  PLAY_COUNT_DEFAULT  0xFF

/**
 * PmAudioManager Result message
 */
typedef std::string   AmResultMessage;

typedef std::string   FilePath;

/**
 *  Possible values:
 *     Default - "" [Empty string]
 *  Possible values when status code is - REQUEST_FAILURE, are:
 *     "audio allocation failure",
 *     "ecnr failure"
 *     "gstreamer error"
 *     "audio deallocation failure"
 *     "audio mute failure"
 *     "audio demute failure"
 *  Possible values when the status code is - AUDIO_STATE_CHANGE, are:
 *     ""  [By default empty when an AUDIO_STATE_CHANGE happens]
 *     "streaming stopped due to external trigger",
 *     "streaming paused due to external trigger"
 */
typedef std::string   StatusMessage;

typedef bool          MuteState;

typedef unsigned char PlayCount;

typedef unsigned char StreamingState;

typedef unsigned char EventType;

typedef short unsigned int GeniviAMSinkId;

typedef short int          GeniviAMPropertyType;

typedef short int          GeniviAMVolumeLevel;

typedef std::map<GeniviAMPropertyType, GeniviAMVolumeLevel> AmPropertyType2VolumeMap;

/**
 * AmSessionId will be created and maintained for a particular session by AM
 * and will be valid until the corresponding session is closed(By invoking the stop Audio)
 */
typedef unsigned int AmSessionId;

typedef unsigned long long int AmActToken;

/**
 * SPM System State
 */
typedef enum
{
   AM_SPM_SYSTEM_OFF = 0UL,
   AM_SPM_SYSTEM_ON,
   AM_SPM_SYSTEM_STANDBY
} AmState;

/**
 * PmAudioaManager result code
 */
typedef enum
{
   AM_RESULT_OK = 0UL,
   AM_RESULT_ERR_GENERAL,
   AM_RESULT_ERR_NOT_IMPLEMENTED,
   AM_RESULT_ERR_INVALID_PARAMETER,
   AM_RESULT_ERR_TIMEOUT
} AmResultCode;

/**
 * Default value for the generated Am Session ID.
 */
enum
{
   AM_SESSION_ID_DEFAULT = 0UL,
   AM_SESSION_ID_CLIENTREQUEST
};

typedef enum
{
   AM_UNKNOWN = 0UL,     /**< Unknown source, default value*/
   AM_VEHICLERINGTONE,   /**< For playing the vehicle ringtone or the playable audio file */
   AM_PHONEAUDIO,        /**< For phone call audio streaming */
   AM_WAITINGMODE,       /**< For waiting mode tone to be streamed for the far end device */
   AM_VOICERECOGNITION,  /**< Stream the voice recognition audio */
   AM_ALERTTONE          /**< For the Alert tone to be played */
}AudioChannel;

typedef enum{
   UNKNOWN = 0UL,
   NARROWBAND,
   WIDEBAND,
   NARROWBAND_SIRI,
   WIDEBAND_SIRI
}SamplingType;

typedef enum{
   NOT_ALLOCATED = 0UL,
   PREPARING,
   READY_TO_STREAM,
   STREAMING,
   STREAMING_PAUSED,
   DEALLOCATING,
   FAILURE = 255UL
}RouteState;

typedef enum{
   NOT_PLAYING = 0UL, /**< Audio is not played [DEFAULT] */
   PLAYING ,          /**< Either audio files are played / when the audio streaming is on */
   MUTED,             /**< Audio is muted [Happens only based on the user request] */
   DEMUTED,           /**< Audio is Demuted [Happens only based on the user request] */
   STOPPED,           /**< Either the audio files are stopped playing / when the audio streaming is stopped */
   PAUSED             /**< Audio streaming is paused [Happens only based on the trigger from ARL] */
}StreamingStateEnumType;

typedef enum
{
   AM_IF_MSG_ID_START = 0x00u,

   //Method Request
   AM_IF_MSG_ID_PREPARE_AUDIO_ROUTE,
   AM_IF_MSG_ID_PLAY_AUDIO_FILE,
   AM_IF_MSG_ID_PLAY_HF_AUDIO,
   AM_IF_MSG_ID_STOP_AUDIO,
   AM_IF_MSG_ID_PAUSE_AUDIO,
   AM_IF_MSG_ID_SET_MIC_MUTE_STATE,
   AM_IF_MSG_ID_GET_AUDIO_SINK_VOLUME,
   AM_IF_MSG_ID_SET_AUDIO_SINK_VOLUME,

   //Method Result
   AM_IF_MSG_ID_ECNR_INITIALIZE_RESULT,
   AM_IF_MSG_ID_ECNR_DESTROY_RESULT,
   AM_IF_MSG_ID_ECNR_SET_CONFIGURATION_RESULT,
   AM_IF_MSG_ID_ECNR_SET_SEND_MUTE_SWITCH_RESULT,
   AM_IF_MSG_ID_ECNR_START_AUDIO_RESULT,
   AM_IF_MSG_ID_ECNR_STOP_AUDIO_RESULT,
   AM_IF_MSG_ID_ECNR_GET_VERSION_RESULT,
   AM_IF_MSG_ID_REQUEST_AUDIO_ROUTE_RESULT,
   AM_IF_MSG_ID_RELEASE_AUDIO_ROUTE_RESULT,
   AM_IF_MSG_ID_SOURCE_ACTIVITY_RESULT,
   AM_IF_MSG_ID_REQUEST_MUTE_ACTION_RESULT,
   AM_IF_MSG_ID_PLAYBACK_STATUS,
   AM_IF_MSG_ID_GAM_GET_LIST_MAIN_SINK_SND_PROP_RESULT,
   AM_IF_MSG_ID_GAM_SET_MAIN_SINK_SND_PROP_RESULT,
   AM_IF_MSG_ID_GAM_UPDATE_AUDIO_SINK_VOL_LIST,

   AM_IF_MSG_ID_END
}AmIfMsgId;

typedef enum
{
   REQUEST_FAILURE = 0UL,
   AUDIO_STATE_CHANGE,
   SESSION_REMOVED
}EventTypeEnumType;

typedef enum
{
   RING_TONE = 0UL,
   WAITINGMODE_TONE
}ToneType;

} // namespace pmaudiomanager

#endif // PmAudioManagerTypesInternal_h
