/**
 * @file PmAudioManagerRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the PmAudioManagerRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PmAudioManager.
 *
 * @ingroup PmAudioManager
 */

#ifndef PmAudioManagerRequestIf_h
#define PmAudioManagerRequestIf_h

#include "PmSingleton.h"
#include "IPmAudioManagerRequestIf.h"
#include "LoopbackData.h"

namespace pmaudiomanager
{

/**
 * PmAudioManagerRequestIf class definition
 */
class PmAudioManagerRequestIf final : public IPmAudioManagerRequestIf, public PmSingleton <PmAudioManagerRequestIf>
{
public:

   /**
    * This interface is used to instantiate and initialize all PM Audiomanager related components.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult intializePmAudioManager(ahl_tclBaseOneThreadApp* mainApplication) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] IPmAudioManagerCallbackIf* - Pointer to the callback interface of PMAudiomanager
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult registerPmAudioManagerCallbackIf(IN const IPmAudioManagerCallbackIf* iPmAudioManagerCallbackIf) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmState amState- Provides the SPM system state / BT System state to AM
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void setAmState(IN const AmState amState) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in]  AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *             AudioChannel - Start the audio streaming based on the requested audio channel.
    *
    *             Note:- When the value of the parameter - "AmSessionId" sent as part of the request is "0",
    *                the request shall be treated as a new session by AM. AM would start the new audio session &
    *                the newly generated AmSessionId shall be informed to PmCore.
    *                The ID shall be used for subsequent communication (requests / notifications)
    *                between PmCore and AM in the same session.
    *                The ID shall be valid till the audio is released / stopped.
    *
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult prepareAudioRoute(IN const AmSessionId amSessionId, IN const AudioChannel audioChannelId) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in]  AmSessionId - An unique ID for an audio session.
    *                The same ID shall be used for subsequent communication (requests / notifications)
    *                between PmCore and AM in the same session.
    *            FilePath - Valid file path of the audio file that has to be played
    *            PlayCount- Denotes the number of times the file has to be played.
    *                This parameter shall be set to FF (Play repeatedly)
    *                - when the vehicle ringtone for an incoming call has to be played [OR]
    *                - when the waiting mode tone has to be played
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult playAudio(IN const AmSessionId amSessionId, IN const FilePath& filePath,
         IN const PlayCount playCount, IN const ToneType toneType) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            SamplingType - NARROWBAND or WIDEBAND -Denotes the sampling type.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult playAudio(IN const AmSessionId amSessionId, IN const SamplingType samplingType) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult stopAudio(IN const AmSessionId amSessionId) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] MuteState muteState - MUte state operation to be performed.
    *               0x00 - unmute, 0x01 - mute
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult setMicMuteState(IN const MuteState muteState) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult pauseAudio(IN const AmSessionId amSessionId) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            GeniviAMSinkID - Sink ID
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult getAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId) override;

   /**
    * overridden from IPmAudioManagerRequestIf
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            GeniviAMSinkID - Sink ID
    *            AmPropertyType2VolumeMap - Property Value
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   AmResult setAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyVal) override;

   /**
    * Loopback method invoked from the entry thread context
    *
    * @param[in] AmActToken - Acknowledgment token for the sent loopback message
    * @param[in] LoopbackData* - pointer to the loopback message.
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void onLoopbackMessage(IN const AmActToken act, IN com::bosch::pmcommon::LoopbackData* loopbackData);

private:
   friend class PmSingleton<PmAudioManagerRequestIf>;

   /**
    * Constructor of PmAudioManagerRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   PmAudioManagerRequestIf();

   /**
    * Destructor of PmAudioManagerRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~PmAudioManagerRequestIf();
};
}

#endif //PmAudioManagerRequestIf_h
