/**
 * @file IPmAudioManagerRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the IPmAudioManagerRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of PmAudioManager.
 *          Note: 
 *          TODO: Need to check the need for Act Token to be added as an input parameter for the methods.
 *
 * @ingroup PmAudioManager
 */

#ifndef IPmAudioManagerRequestIf_h
#define IPmAudioManagerRequestIf_h

#include "PmAudioManagerIfTypes.h"

//class forward declaration
class ahl_tclBaseOneThreadApp;

namespace pmaudiomanager
{
/**
 * forward declaration
 */
class IPmAudioManagerCallbackIf;

/**
 * IPmAudioManagerRequestIf class definition
 */
class IPmAudioManagerRequestIf
{
public:
   /**
    * Destructor of IPmAudioManagerRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~IPmAudioManagerRequestIf() {}

   /**
    * This interface is used to instantiate and initialize all PM Audiomanager related components.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult intializePmAudioManager(ahl_tclBaseOneThreadApp* mainApplication) = 0;

   /**
    * This API is used to register the callback interface of PmAudioManager with the Request interface of
    * PMAudiomanager.
    * This API should be invoked on receiving a successful response for the API- "intializePmAudioManager".
    *
    * @param[in] IPmAudioManagerCallbackIf* - Pointer to the callback interface of PMAudiomanager
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult registerPmAudioManagerCallbackIf(IN const IPmAudioManagerCallbackIf* iPmAudioManagerCallbackIf) = 0;

   /**
    * This API is used to set the state of the PM Audio-manager.
    * Application state (or) BT state has to be concluded after discussions with CpM.
    * To check with ASF documentation on detail.
    *
    * @param[in] AmState amState- Provides the SPM system state / BT System state to AM
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual void setAmState(IN const AmState amState) = 0;

   /**
    * This interface is used to (only) allocate the audio route for the requested audio channel ID.
    *
    * @param[in]  AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *             AudioChannel - Start the audio streaming based on the requested audio channel.
    *
    *             Note:- When the value of the parameter - "AmSessionId" sent as part of the request is "0",
    *                the request shall be treated as a new session by AM. AM would start the new audio session &
    *                the newly generated AmSessionId shall be informed to PmCore.
    *                The ID shall be used for subsequent communication (requests / notifications)
    *                between PmCore and AM in the same session.
    *                The ID shall be valid till the audio is released / stopped.
    *
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult prepareAudioRoute(IN const AmSessionId amSessionId, IN const AudioChannel audioChannelId) = 0;

   /**
    * This interface is used to play a valid playable file.
    *
    * @param[in]  AmSessionId - An unique ID for an audio session.
    *                The same ID shall be used for subsequent communication (requests / notifications)
    *                between PmCore and AM in the same session.
    *            FilePath - Valid file path of the audio file that has to be played
    *            PlayCount- Denotes the number of times the file has to be played.
    *                This parameter shall be set to FF (Play repeatedly)
    *                - when the vehicle ringtone for an incoming call has to be played [OR]
    *                - when the waiting mode tone has to be played
    *             toneType - Specifies the type of tone- Ring-tone or Waiting mode tone
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult playAudio(IN const AmSessionId amSessionId, IN const FilePath& filePath,
         IN const PlayCount playCount, IN const ToneType toneType) = 0;

   /**
    * This interface allows the client to stream the Phone call Audio, waiting mode tone, alert tone
    * or VoiceRecognition audio.
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            SamplingType - NARROWBAND or WIDEBAND -Denotes the sampling type.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult playAudio(IN const AmSessionId amSessionId, IN const SamplingType samplingType) = 0;

   /**
    * This interface is used to stop the playing of file triggered by the playFile request.
    *    1. AmSessionId will become invalid once the stopFile is processed successfully.
    *    2. In case of Incoming call scenario, after accepting the call,
    *    instead of invoking this method, the method "StartAudio" shall be invoked with the appropriate channel ID.
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult stopAudio(IN const AmSessionId amSessionId) = 0;

   /**
    * The interface shall be invoked to mute / unmute the microphone in the Head Unit.
    * Note: The request shall be processed, only if an ECNR session is running in the context of PM Application.
    *
    * @param[in] MuteState muteState - MUte state operation to be performed.
    *               0x00 - unmute, 0x01 - mute
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult setMicMuteState(IN const MuteState muteState) = 0;

   /**
    * This interface shall be triggered to pause the audio that is played.
    * This interface shall not relinquish the Audio channel.
    * The audio play shall be resumed by invoking the playAudio interface.
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult pauseAudio(IN const AmSessionId amSessionId) = 0;

   /**
    * This interface is used to get the volume Properties of the specified sink
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            GeniviAMSinkID - Sink ID
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult getAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId) = 0;

   /**
    * This interface is used to set the volume Properties of the specified sink
    *
    * @param[in] AmSessionId - An unique ID for an audio session.
    *               The same ID shall be used for subsequent communication (requests / notifications) between
    *               PmCore and AM in the same session.
    *            GeniviAMSinkID - Sink ID
    *            AmPropertyType2VolumeMap - Property Value
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult- Result of the operation
    */
   virtual AmResult setAudioSinkVolume(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyVal) = 0;
};
}

#endif //IPmAudioManagerRequestIf_h
