/**
 * @file GeniviAmRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for Genivi Audio Manager DBUS Proxy
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "GeniviAmRequestIf.h"
#include "IGeniviAmCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/GeniviAmRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

using namespace ::org::genivi::audiomanager::CommandInterface;
using namespace ::asf::core;

namespace pmaudiomanager {

GeniviAmRequestIf::GeniviAmRequestIf() : IGeniviAmRequestIf(false)
{
   ETG_TRACE_USR4(("GeniviAmRequestIf(): Constructor"));

   _callBackIf = nullptr;
}

GeniviAmRequestIf::~GeniviAmRequestIf()
{
   ETG_TRACE_USR4(("GeniviAmRequestIf(): Destructor"));

   destroyProxy();

   _callBackIf = nullptr;
}

void GeniviAmRequestIf::createProxy(void)
{
   ETG_TRACE_USR4(("createProxy(): GeniviAm"));

   if(nullptr != _callBackIf)
   {
      _cmdIfProxy = CommandInterfaceProxy::createProxy("geniviAmCmdIfPort", *this);
   }
}

void GeniviAmRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR4(("destroyProxy(): GeniviAm"));

   if(nullptr != _cmdIfProxy.get())
   {
      _cmdIfProxy.reset();
   }
}

void GeniviAmRequestIf::setCallbackIf(IGeniviAmCallBackIf* callBackIf)
{
   _callBackIf = callBackIf;

   createProxy();
}

void GeniviAmRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("onAvailable(): GeniviAm"));

   if((nullptr != _cmdIfProxy.get()) && (proxy == _cmdIfProxy))
   {
      if(nullptr != _callBackIf)
      {
         _callBackIf->onServiceProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
               _cmdIfProxy->getDBusObjectPath());

         registerProperties();
      }

      _cmdIfAvailability = true;
   }
}

void GeniviAmRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("onUnavailable(): GeniviAm"));

   if((nullptr != _cmdIfProxy.get()) && (proxy == _cmdIfProxy))
   {
      if(nullptr != _callBackIf)
      {
         _callBackIf->onServiceProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
               _cmdIfProxy->getDBusObjectPath());
      }

      deRegisterProperties();

      _cmdIfAvailability = false;
   }
}

act_t GeniviAmRequestIf::sendGetMainSinkSoundPropertiesList(const GeniviAMSinkId sinkId)
{
   ETG_TRACE_USR4(("sendGetMainSinkSoundPropertiesList(): GeniviAm"));

   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_cmdIfAvailability && (nullptr != _callBackIf) && (nullptr != _cmdIfProxy.get()))
   {
      act = _cmdIfProxy->sendGetListMainSinkSoundPropertiesRequest(*_callBackIf, sinkId);
   }

   return act;
}

act_t GeniviAmRequestIf::sendSetMainSinkSoundProperty(const GeniviAMSinkId sinkId, const GeniviAMPropertyType propertyType, const GeniviAMVolumeLevel volumeLevel)
{
   ETG_TRACE_USR4(("sendSetMainSinkSoundProperty(): GeniviAm"));

   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_cmdIfAvailability && (nullptr != _callBackIf) && (nullptr != _cmdIfProxy.get()))
   {
      act = _cmdIfProxy->sendSetMainSinkSoundPropertyRequest(*_callBackIf, sinkId, SetMainSinkSoundPropertyRequestSoundPropertyStruct(propertyType, volumeLevel));
   }

   return act;
}

void GeniviAmRequestIf::registerProperties()
{
   ETG_TRACE_USR4(("GeniviAmRequestIf : registerProperties() entered"));

   _cmdIfProxy->sendMainSinkSoundPropertyChangedRegister(*_callBackIf);
}

void GeniviAmRequestIf::deRegisterProperties()
{
   ETG_TRACE_USR4(("GeniviAmRequestIf : deRegisterProperties() entered"));

   _cmdIfProxy->sendMainSinkSoundPropertyChangedDeregisterAll();
}

}
