/**
 * @file GeniviAmWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the GeniviAm methods and response from GeniviAm is forwarded to AmMainController
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef GeniviAmWrapper_h
#define GeniviAmWrapper_h

#include "PmAudioManagerIfTypes.h"
#include "GeniviAmTypes.h"
#include "CcDbusIfTypes.h"

namespace pmaudiomanager {

// class forward declarations
class IGeniviAmCallBackIf;
class IGeniviAmRequestIf;

/**
 * GeniviAm Wrapper
 */
class GeniviAmWrapper final
{
public:
    /**
    * Constructor of GeniviAmWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
    GeniviAmWrapper();

    /**
    * Parameterised Constructor of GeniviAmWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
    GeniviAmWrapper(IGeniviAmCallBackIf* geniviAmCallbackIf, IGeniviAmRequestIf* geniviamRequestIf);

    /**
    * Destructor of GeniviAmWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
    virtual ~GeniviAmWrapper();

    /**
    * Copy constructor of GeniviAmWrapper class
    *
    * @param[in]    ref     reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return GeniviAmWrapper
    */
    GeniviAmWrapper(IN const GeniviAmWrapper& ref);

    /**
    * Assignment operator of GeniviAmWrapper class
    *
    * @param[in]    ref     reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return GeniviAmWrapper
    */
    GeniviAmWrapper& operator=(IN const GeniviAmWrapper& ref);

    /**
    * getMainSinkSoundPropertiesList
    *
    * @param[in]    sinkId      Sink ID
    * @param[in]    amSessionId AmSessionId
    * @param[in,out]
    *
    * @return AmErrorCode
    */
    AmErrorCode getMainSinkSoundPropertiesList(const AmSessionId amSessionId, const GeniviAMSinkId sinkId);

    /**
    * getListMainSinkSoundPropertiesListCB
    *
    * @param[in]    responseStatus   Error from getMainSinkSoundPropertiesList method call
    * @param[in]    act              act token
    * @param[in,out]
    *
    * @return
    */
    void getListMainSinkSoundPropertiesCB(const GeniviAMResult responseStatus, const act_t act);

    /**
    * getListMainSinkSoundPropertiesListCB
    *
    * @param[in]    propertyValue    Property vector from getMainSinkSoundPropertiesList method call
    * @param[in]    act              act token
    * @param[in,out]
    *
    * @return
    */
    void getListMainSinkSoundPropertiesCB(const AmPropertyType2VolumeMap propertyValue, const act_t act);

    /**
    * setMainSinkSoundProperty
    *
    * @param[in]    sinkId            Sink ID
    * @param[in]    volumeLevel       Volume Level to set
    * @param[in]    propertyType      Property Type
    * @param[in]    amSessionId       AmSessionId
    * @param[in,out]
    *
    * @return AmErrorCode
    */
    AmErrorCode setMainSinkSoundProperty(const AmSessionId sessionId, const GeniviAMSinkId sinkId,
            const GeniviAMPropertyType propertyType, const GeniviAMVolumeLevel volumeLevel);

    /**
    * setMainSinkSoundPropertyCB
    *
    * @param[in]    responseStatus   Error from getMainSinkSoundPropertiesList method call
    * @param[in]    act              act token
    * @param[in,out]
    *
    * @return
    */
    void setMainSinkSoundPropertyCB(const GeniviAMResult responseStatus, const act_t act);

    /**
    * onMainSinkSoundPropertyChanged
    *
    * @param[in]    sinkId          Sink ID
    * @param[in]    propertyType    Property Type
    * @param[in]    volumeLevel     Volume Level value
    * @param[in,out]
    *
    * @return
    */
    void onMainSinkSoundPropertyChanged(const GeniviAMSinkId sinkId, const GeniviAMPropertyType propertyType,
            const GeniviAMVolumeLevel volumeLevel);

private:
   /**
    * This method is used to store the mapping between dbus act and AmSessionId
    *
    * @param[in] dbusAct: dbus asynchronous completion token
    * @param[in] amSessionId: am session id
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void mapDbusActToAmSessionId(IN const act_t dbusAct, IN const AmSessionId amSessionId);

   /**
    * This method is used to retrieve the AmSessionId from dbus act
    *
    * @param[in] dbusAct: dbus asynchronous completion token
    * @param[out] amSessionId: am session id
    * @param[in,out]
    *
    * @return void
    */
   void retrieveAmSessionIdFromDbusAct(IN const act_t dbusAct, OUT AmSessionId& amSessionId);

   /**
    * This method is used to clear the mapping between dbus act and AmSessionId
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void clearDbusActToAmSessionIdMap();

   IGeniviAmRequestIf* _geniviAMRequestIf; /**< pointer to GeniviAm Request */
   IGeniviAmCallBackIf* _geniviAMCallBackIf; /**< pointer to GeniviAm Callback */
   std::map<act_t, AmSessionId> _actToSessionIdMap; /**< map between act and AmSessionId */
};

} //pmaudiomanager

#endif  //GeniviAmWrapper_h
