/**
 * @file AudioPlayerController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the wrapper functions of AudioPlayer requests and response methods.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef AudioPlayerController_h
#define AudioPlayerController_h

#include "PmAudioManagerIfTypes.h"
#include "AudioPlayerTypes.h"
#include "IAudioPlayer.h"

namespace pmaudiomanager {

   //class forward declaration
   class AudioPlayerWrapper;

/**
 * Audio Player Controller class.
 */
class AudioPlayerController final
{
public:
   /**
    * Constructor of AudioPlayerController class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   AudioPlayerController(AudioPlayerWrapper*);

   /**
    * Destructor of AudioPlayerController class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   virtual ~AudioPlayerController();

   /**
    * Copy constructor => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   AudioPlayerController(IN const AudioPlayerController& ref);

   /**
    * Assignment operator => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return reference to AudioPlayerController
    */
   AudioPlayerController& operator=(IN const AudioPlayerController& ref);

   /**
    * This method is used to play the given audio file
    *
    * @param[in] sessionId: Session Id
    * @param[in] filePath: audio file path
    * @param[in] playCount: play count
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode playAudioFileRequest(IN const AmSessionId sessionId,
         IN const FilePath& filePath, IN const PlayCount playCount = PLAY_COUNT_DEFAULT);

   /**
    * This method is used to stop the play back which is in progress
    *
    * @param[in] sessionId: Session Id
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode stopPlaybackRequest(IN const AmSessionId sessionId);

   /**
    * This method is used to set the ALSA device
    *
    * @param[in] sessionId: Session Id
    * @param[in] sink   : Audio sink
    * @param[in] source : Audio source
    * @param[in] toneType : Type of tone
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode setAlsaDeviceRequest(IN const AmSessionId sessionId,
         IN const AudioSink& sink, IN const AudioSource& source, IN const ToneType toneType);

   /**
    * This method is used to destroy the AudioPlayer instance for the requested session ID
    *
    * @param[in] sessionId: Session Id
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode destroyAudioPlayer(IN const AmSessionId sessionId);

private:
   AudioPlayerWrapper* _audioPlayerWrapper;  /**< Instance of the AudioPlayerWrapper class */

   // AmSession id is mapped to the pointer of AudioPlayer interface
   std::map <AmSessionId, IAudioPlayer*> _sessionIdAudioPlayerMap;
};

} //pmaudiomanager

#endif // AudioPlayerController_h
