/**
 * @file AudioRouteHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the ARL methods and response from ARL is forwarded to ArlWrapper
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "AudioRouteHandler.h"
#include "ArlWrapper.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AudioRouteHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager {

AudioRouteHandler::AudioRouteHandler() :
      arl_tclISource_ASF(nullptr)
{
   ETG_TRACE_USR4(("AudioRouteHandler: Constructor"));

   _arlWrapper = nullptr;
   _timerPhoneAudio = nullptr;
   _timerVehicleRingtone = nullptr;
   _timerWaitingModeTone = nullptr;
   _timerVoiceRecognition = nullptr;
}

AudioRouteHandler::AudioRouteHandler(ArlWrapper* arlWrapper, ahl_tclBaseOneThreadApp* mainApplication) :
      arl_tclISource_ASF(mainApplication)
{
   ETG_TRACE_USR4(("AudioRouteHandler: Parameterized Constructor"));

   _arlWrapper = arlWrapper;

   ms_t timeout = (ms_t)com::bosch::pmcommon::PmConfiguration::getInstance().getWaitTimeForArlResponse();

   _timerPhoneAudio = new com::bosch::pmcommon::AsfTimer<AudioRouteHandler, std::string>();
   if(nullptr != _timerPhoneAudio)
   {
      com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> timerData("PhoneAudio",
            timeout, TIMER_REPEAT_COUNT);

      _timerPhoneAudio->configure(this, timerData);
   }

   _timerVehicleRingtone = new com::bosch::pmcommon::AsfTimer<AudioRouteHandler, std::string>();
   if(nullptr != _timerVehicleRingtone)
   {
      com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> timerData("VehicleRingtone",
            timeout, TIMER_REPEAT_COUNT);

      _timerVehicleRingtone->configure(this, timerData);
   }

   _timerWaitingModeTone = new com::bosch::pmcommon::AsfTimer<AudioRouteHandler, std::string>();
   if(nullptr != _timerWaitingModeTone)
   {
      com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> timerData("WaitingModeTone",
            timeout, TIMER_REPEAT_COUNT);

      _timerWaitingModeTone->configure(this, timerData);
   }

   _timerVoiceRecognition = new com::bosch::pmcommon::AsfTimer<AudioRouteHandler, std::string>();
   if(nullptr != _timerVoiceRecognition)
   {
      com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> timerData("VoiceRecognition",
            timeout, TIMER_REPEAT_COUNT);

      _timerVoiceRecognition->configure(this, timerData);
   }

}

AudioRouteHandler::~AudioRouteHandler()
{
   ETG_TRACE_USR4(("AudioRouteHandler: Destructor"));

   _arlWrapper = nullptr;

   if(nullptr != _timerPhoneAudio)
   {
      delete _timerPhoneAudio;
      _timerPhoneAudio = nullptr;
   }

   if(nullptr != _timerVehicleRingtone)
   {
      delete _timerVehicleRingtone;
      _timerVehicleRingtone = nullptr;
   }

   if(nullptr != _timerWaitingModeTone)
   {
      delete _timerWaitingModeTone;
      _timerWaitingModeTone = nullptr;
   }

   if(nullptr != _timerVoiceRecognition)
   {
      delete _timerVoiceRecognition;
      _timerVoiceRecognition = nullptr;
   }
}

AmResultCode AudioRouteHandler::setSourceAvailability(const SrcAvailability srcAvailability,
      const AvailabilityReason availabilityReason, const AudioChannel channel, const SubSource subSource)
{
   ETG_TRACE_USR4(("setSourceAvailability() entered"));

   AmResultCode result = AM_RESULT_ERR_GENERAL;
   arl_tenSource enSrcNum = ARL_SRC_NONE;

   convertAudioChannelToArlSourceNum(channel, enSrcNum);

   if(ARL_SRC_NONE != enSrcNum)
   {
      arl_tenSrcAvailability arlSrcAvailability = ARL_EN_SRC_NOT_AVAILABLE;
      arl_tenAvailabilityReason arlAvailabilityReason = ARL_EN_REASON_NEWMEDIA;

      convertSrcAvailabilityToArlSrcAvailability(srcAvailability, arlSrcAvailability);
      convertAvailabilityReasonToArlAvailabilityReason(availabilityReason, arlAvailabilityReason);

      if(true == bSetSourceAvailability(arlSrcAvailability, arlAvailabilityReason, enSrcNum, subSource))
      {
         ETG_TRACE_USR4(("setSourceAvailability: Success"));

         result = AM_RESULT_OK;
      }
      else
      {
         ETG_TRACE_ERR(("setSourceAvailability: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("setSourceAvailability: Invalid Source"));
   }

   return result;
}

AmResultCode AudioRouteHandler::sendSourceActivityResult(const AudioChannel channel, const SrcActivity srcActivity)
{
   ETG_TRACE_USR4(("sendSourceActivityResult() entered"));

   AmResultCode result = AM_RESULT_ERR_GENERAL;
   arl_tenSource enSrcNum = ARL_SRC_NONE;

   convertAudioChannelToArlSourceNum(channel, enSrcNum);

   if(ARL_SRC_NONE != enSrcNum)
   {
      arl_tSrcActivity arlSrcActivity;

      convertSrcActivityToArlSrcActivity(srcActivity, arlSrcActivity);

      vSourceActivityResult(enSrcNum, arlSrcActivity);

      ETG_TRACE_USR4(("sendSourceActivityResult: Success"));

      result = AM_RESULT_OK;
   }
   else
   {
      ETG_TRACE_ERR(("sendSourceActivityResult: Invalid Source"));
   }

   return result;
}

AmResultCode AudioRouteHandler::sendAudioRouteRequest(const AudioChannel channel)
{
   ETG_TRACE_USR4(("sendAudioRouteRequest() entered"));

   AmResultCode result = AM_RESULT_ERR_GENERAL;
   arl_tenSource enSrcNum = ARL_SRC_NONE;

   convertAudioChannelToArlSourceNum(channel, enSrcNum);

   if(ARL_SRC_NONE != enSrcNum)
   {
      vSetSrcAvailable(TRUE);

      if((ARL_SRC_PHONE_OUTBAND_RING == enSrcNum) || (ARL_SRC_PHONE_VR == enSrcNum))
      {
         if(true == bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_ON))
         {
            ETG_TRACE_USR4(("sendAudioRouteRequest: Success"));

            startTimer(channel);

            result = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("sendAudioRouteRequest: Failed"));
         }
      }
      else
      {
         if(true == bSetAudioRouteRequest(SourceID(enSrcNum, SUB_SOURCE_1), ARL_EN_ISRC_ACT_ON))
         {
            ETG_TRACE_USR4(("sendAudioRouteRequest: Success"));

            startTimer(channel);

            result = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("sendAudioRouteRequest: Failed"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("sendAudioRouteRequest: Invalid Source"));
   }

   return result;
}

AmResultCode AudioRouteHandler::sendAudioReleaseRequest(const AudioChannel channel)
{
   ETG_TRACE_USR4(("sendAudioReleaseRequest() entered"));

   AmResultCode result = AM_RESULT_ERR_GENERAL;
   arl_tenSource enSrcNum = ARL_SRC_NONE;

   convertAudioChannelToArlSourceNum(channel, enSrcNum);

   if(ARL_SRC_NONE != enSrcNum)
   {
      if((ARL_SRC_PHONE_OUTBAND_RING == enSrcNum) || (ARL_SRC_PHONE_VR == enSrcNum))
      {
         if(true == bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_OFF))
         {
            ETG_TRACE_USR4(("sendAudioReleaseRequest: Success"));

            stopTimer(channel);

            result = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("sendAudioReleaseRequest: Failed"));
         }
      }
      else
      {
         if(true == bSetAudioRouteRequest(SourceID(enSrcNum, SUB_SOURCE_1), ARL_EN_ISRC_ACT_OFF))
         {
            ETG_TRACE_USR4(("sendAudioReleaseRequest: Success"));

            stopTimer(channel);

            result = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("sendAudioReleaseRequest: Failed"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("sendAudioReleaseRequest: Invalid Source"));
   }

   return result;
}

AmResultCode AudioRouteHandler::setMuteStateAction(const AudioChannel channel, const MuteState muteAction,
      const MuteReason muteReason)
{
   ETG_TRACE_USR4(("setMuteStateAction() entered"));

   AmResultCode result = AM_RESULT_ERR_GENERAL;
   arl_tenSource enSrcNum = ARL_SRC_NONE;

   convertAudioChannelToArlSourceNum(channel, enSrcNum);

   if(ARL_SRC_NONE != enSrcNum)
   {
      arl_tenMuteAction arlMuteAction = ARL_EN_MUTEACTION_DEMUTE;
      audioroutinglib::arl_tclBitmask<arl_enMuteReason> arlMuteReason = ARL_EN_MUTE_REASON_OTHER;

      convertMuteActionToArlMuteAction(muteAction, arlMuteAction);
      convertMuteReasonToArlMuteReason(muteReason, arlMuteReason);

      if(true == bSetMuteState(enSrcNum, arlMuteAction, arlMuteReason))
      {
         ETG_TRACE_USR4(("setMuteStateAction: Success"));

         result = AM_RESULT_OK;
      }
      else
      {
         ETG_TRACE_ERR(("setMuteStateAction: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("setMuteStateAction: Invalid Source"));
   }

   return result;
}

tBool AudioRouteHandler::bOnSrcActivity(arl_tenSource enSrcNum, tU16 subSource, const arl_tSrcActivity& rfcoSrcActivity)
{
   ETG_TRACE_USR4(("bOnSrcActivity() entered"));

   tBool ret = FALSE;

   if(nullptr != _arlWrapper)
   {
      AudioChannel channel = AM_UNKNOWN;

      convertArlSourceNumToAudioChannel(enSrcNum, channel);

      if(AM_UNKNOWN != channel)
      {
         SrcActivity srcActivity = SRC_ACTIVITY_OFF;

         convertArlSrcActivityToSrcActivity(rfcoSrcActivity, srcActivity);

         if(SRC_ACTIVITY_ON == srcActivity)
         {
            stopTimer(channel);
         }

         ret = _arlWrapper->onSourceActivityCB(channel, subSource, srcActivity);
      }
      else
      {
         ETG_TRACE_ERR(("bOnSrcActivity: Invalid Source"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("bOnSrcActivity: ArlWrapper NULL"));
   }

   return ret;
}

tBool AudioRouteHandler::bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& rfcoAllocRoute)
{
   ETG_TRACE_USR4(("bOnAllocate() entered"));

   tBool ret = FALSE;

   if(nullptr != _arlWrapper)
   {
      AudioChannel channel = AM_UNKNOWN;

      convertArlSourceNumToAudioChannel(enSrcNum, channel);

      if(AM_UNKNOWN != channel)
      {
         AmResult amResult(AM_RESULT_OK, "");
         AudioDeviceList audioSink;
         AudioDeviceList audioSource;

         //retrieve audioSink and audioSource
         for(size_t i = 0; i < rfcoAllocRoute.listInputDev.strALSADev.size(); i++)
         {
            audioSource.push_back(rfcoAllocRoute.listInputDev.strALSADev[i].szValue);

            ETG_TRACE_USR4(("bOnAllocate: AudioSource = \"%50s\" ", rfcoAllocRoute.listInputDev.strALSADev[i].szValue));
         }

         for(size_t j = 0; j < rfcoAllocRoute.listOutputDev.strALSADev.size(); j++)
         {
            audioSink.push_back(rfcoAllocRoute.listOutputDev.strALSADev[j].szValue);

            ETG_TRACE_USR4(("bOnAllocate: AudioSink = \"%50s\" ", rfcoAllocRoute.listOutputDev.strALSADev[j].szValue));
         }

         ret = _arlWrapper->requestAudioRouteCB(amResult, channel, audioSink, audioSource);
      }
      else
      {
         ETG_TRACE_ERR(("bOnAllocate: Invalid Source"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("bOnAllocate: ArlWrapper NULL"));
   }

   return ret;
}

tBool AudioRouteHandler::bOnDeAllocate(arl_tenSource enSrcNum)
{
   ETG_TRACE_USR4(("bOnDeAllocate() entered"));

   tBool ret = FALSE;

   if(nullptr != _arlWrapper)
   {
      AudioChannel channel = AM_UNKNOWN;

      convertArlSourceNumToAudioChannel(enSrcNum, channel);

      if(AM_UNKNOWN != channel)
      {
         AmResult amResult(AM_RESULT_OK, "");

         ret = _arlWrapper->releaseAudioRouteCB(amResult, channel);
      }
      else
      {
         ETG_TRACE_ERR(("bOnDeAllocate: Invalid Source"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("bOnDeAllocate: ArlWrapper NULL"));
   }

   return ret;
}

tBool AudioRouteHandler::bOnMuteState(arl_tenSource enSrcNum, arl_tenMuteState enMuteState)
{
   ETG_TRACE_USR4(("bOnMuteState() entered"));

   tBool ret = FALSE;

   if(nullptr != _arlWrapper)
   {
      AudioChannel channel = AM_UNKNOWN;

      convertArlSourceNumToAudioChannel(enSrcNum, channel);

      if(AM_UNKNOWN != channel)
      {
         AmResult amResult(AM_RESULT_OK, "");
         MuteState muteState = false;

         convertArlMuteStateToMuteState(enMuteState, muteState);

         ret = _arlWrapper->requestMuteActionCB(amResult, channel, muteState);
      }
      else
      {
         ETG_TRACE_ERR(("bOnMuteState: Invalid Source"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("bOnMuteState: ArlWrapper NULL"));
   }

   return ret;
}

void AudioRouteHandler::startTimer(const AudioChannel channel)
{
   ETG_TRACE_USR4(("startTimer() entered: %u", ETG_CENUM(AudioChannel, channel)));

   switch(channel)
   {
      case AM_ALERTTONE:
      case AM_PHONEAUDIO:
      {
         if(nullptr != _timerPhoneAudio)
         {
            _timerPhoneAudio->start();
         }
         else
         {
            ETG_TRACE_ERR(("startTimer: timerPhoneAudio NULL"));
         }
      }
      break;
      case AM_VEHICLERINGTONE:
      {
         if(nullptr != _timerVehicleRingtone)
         {
            _timerVehicleRingtone->start();
         }
         else
         {
            ETG_TRACE_ERR(("startTimer: timerVehicleRingtone NULL"));
         }
      }
      break;
      case AM_WAITINGMODE:
      {
         if(nullptr != _timerWaitingModeTone)
         {
            _timerWaitingModeTone->start();
         }
         else
         {
            ETG_TRACE_ERR(("startTimer: timerWaitingModeTone NULL"));
         }
      }
      break;
      case AM_VOICERECOGNITION:
      {
         if(nullptr != _timerVoiceRecognition)
         {
            _timerVoiceRecognition->start();
         }
         else
         {
            ETG_TRACE_ERR(("startTimer: timerVoiceRecognition NULL"));
         }
      }
      break;
      default:
      {
         ETG_TRACE_ERR(("StartTimer called for Unknown channel"));
      }
      break;
   }
}

void AudioRouteHandler::stopTimer(const AudioChannel channel)
{

   ETG_TRACE_USR4(("stopTimer() entered: %u", ETG_CENUM(AudioChannel, channel)));

   switch(channel)
   {
      case AM_ALERTTONE:
      case AM_PHONEAUDIO:
      {
         if((nullptr != _timerPhoneAudio) && (_timerPhoneAudio->isActive()))
         {
            _timerPhoneAudio->stop();
         }
         else
         {
            ETG_TRACE_ERR(("stopTimer: timerPhoneAudio NULL or Not active"));
         }
      }
      break;
      case AM_VEHICLERINGTONE:
      {
         if((nullptr != _timerVehicleRingtone) && (_timerVehicleRingtone->isActive()))
         {
            _timerVehicleRingtone->stop();
         }
         else
         {
            ETG_TRACE_ERR(("stopTimer: timerVehicleRingtone NULL or Not active"));
         }
      }
      break;
      case AM_WAITINGMODE:
      {
         if((nullptr != _timerWaitingModeTone) && (_timerWaitingModeTone->isActive()))
         {
            _timerWaitingModeTone->stop();
         }
         else
         {
            ETG_TRACE_ERR(("stopTimer: timerWaitingModeTone NULL or Not active"));
         }
      }
      break;
      case AM_VOICERECOGNITION:
      {
         if((nullptr != _timerVoiceRecognition) && (_timerVoiceRecognition->isActive()))
         {
            _timerVoiceRecognition->stop();
         }
         else
         {
            ETG_TRACE_ERR(("stopTimer: timerVoiceRecognition NULL or Not active"));
         }
      }
      break;
      default:
      {
         ETG_TRACE_ERR(("stopTimer called for Unknown channel"));
      }
      break;
   }
}

void AudioRouteHandler::timerElapsed(com::bosch::pmcommon::TimerData<std::string, ms_t, RepeatCount> data,
      com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData)
{
   ETG_TRACE_USR4(("AudioRouteHandler::timerElapsed() entered"));
   ETG_TRACE_USR4(("AudioRouteHandler::timerInfo: %s", data._timerInfo.c_str()));

   (void)(timerCallbackData);
   AudioChannel channelId = AM_UNKNOWN;

   if(0 == data._timerInfo.compare("PhoneAudio"))
   {
      channelId = AM_PHONEAUDIO;
   }
   else if(0 == data._timerInfo.compare("VehicleRingtone"))
   {
      channelId = AM_VEHICLERINGTONE;
   }
   else if(0 == data._timerInfo.compare("VoiceRecognition"))
   {
      channelId = AM_VOICERECOGNITION;
   }
   else if(0 == data._timerInfo.compare("WaitingModeTone"))
   {
      channelId = AM_WAITINGMODE;
   }
   else
   {
      //something wrong
   }

   if(AM_UNKNOWN != channelId)
   {
      sendAudioReleaseRequest(channelId);

      if(nullptr != _arlWrapper)
      {
         AmResult amResult(AM_RESULT_ERR_TIMEOUT, "");
         AudioDeviceList audioSink;
         AudioDeviceList audioSource;

         (void)_arlWrapper->requestAudioRouteCB(amResult, channelId, audioSink, audioSource);
      }
   }
}

void AudioRouteHandler::convertAudioChannelToArlSourceNum(const AudioChannel channel, arl_tenSource& enSrcNum)
{
   switch(channel)
   {
      case AM_UNKNOWN:
         enSrcNum = ARL_SRC_NONE;
         break;
      case AM_VEHICLERINGTONE:
         enSrcNum = ARL_SRC_PHONE_OUTBAND_RING;
         break;
      case AM_PHONEAUDIO:
      case AM_ALERTTONE:
         enSrcNum = ARL_SRC_PHONE;
         break;
      case AM_WAITINGMODE:
         enSrcNum = ARL_SRC_PHONE_WAIT;
         break;
      case AM_VOICERECOGNITION:
         enSrcNum = ARL_SRC_PHONE_VR;
         break;
      default:
         enSrcNum = ARL_SRC_NONE;
         break;
   }
}

void AudioRouteHandler::convertSrcAvailabilityToArlSrcAvailability(const SrcAvailability srcAvailability,
      arl_tenSrcAvailability& arlSrcAvailability)
{
   switch(srcAvailability)
   {
      case SRC_NOT_AVAILABLE:
         arlSrcAvailability = ARL_EN_SRC_NOT_AVAILABLE;
         break;
      case SRC_PLAYABLE_UNKNOWN:
         arlSrcAvailability = ARL_EN_SRC_PLAYABLE_UNKNOWN;
         break;
      case SRC_PLAYABLE:
         arlSrcAvailability = ARL_EN_SRC_PLAYABLE;
         break;
      case SRC_NOT_PLAYABLE:
         arlSrcAvailability = ARL_EN_SRC_NOT_PLAYABLE;
         break;
      default:
         arlSrcAvailability = ARL_EN_SRC_NOT_AVAILABLE;
         break;
   }
}

void AudioRouteHandler::convertAvailabilityReasonToArlAvailabilityReason(const AvailabilityReason availabilityReason,
      arl_tenAvailabilityReason& arlAvailabilityReason)
{
   switch(availabilityReason)
   {
      case AVAILABILITY_REASON_NEWMEDIA:
         arlAvailabilityReason = ARL_EN_REASON_NEWMEDIA;
         break;
      case AVAILABILITY_REASON_SAMEMEDIA:
         arlAvailabilityReason = ARL_EN_REASON_SAMEMEDIA;
         break;
      case AVAILABILITY_REASON_NOMEDIA:
         arlAvailabilityReason = ARL_EN_REASON_NOMEDIA;
         break;
      case AVAILABILITY_REASON_TEMPERATURE:
         arlAvailabilityReason = ARL_EN_REASON_TEMPERATURE;
         break;
      case AVAILABILITY_REASON_VOLTAGE:
         arlAvailabilityReason = ARL_EN_REASON_VOLTAGE;
         break;
      case AVAILABILITY_REASON_ERROR:
         arlAvailabilityReason = ARL_EN_REASON_ERROR;
         break;
      case AVAILABILITY_REASON_NO_CONTENT:
         arlAvailabilityReason = ARL_EN_REASON_NO_CONTENT;
         break;
      case AVAILABILITY_REASON_OVERCURRENT:
         arlAvailabilityReason = ARL_EN_REASON_OVERCURRENT;
         break;
      case AVAILABILITY_REASON_INTERNALDISCONNECT:
         arlAvailabilityReason = ARL_EN_REASON_INTERNALDISCONNECT;
         break;
      default:
         arlAvailabilityReason = ARL_EN_REASON_NEWMEDIA;
         break;
   }
}

void AudioRouteHandler::convertSrcActivityToArlSrcActivity(const SrcActivity srcActivity, arl_tSrcActivity& arlSrcActivity)
{
   switch(srcActivity)
   {
      case SRC_ACTIVITY_OFF:
         arlSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
         break;
      case SRC_ACTIVITY_ON:
         arlSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_ON;
         break;
      case SRC_ACTIVITY_PAUSE:
         arlSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE;
         break;
      default:
         arlSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
         break;
   }
}

void AudioRouteHandler::convertMuteActionToArlMuteAction(const MuteState muteAction, arl_tenMuteAction& arlMuteAction)
{
   if(true == muteAction)
   {
      arlMuteAction = ARL_EN_MUTEACTION_MUTE;
   }
   else
   {
      arlMuteAction = ARL_EN_MUTEACTION_DEMUTE;
   }
}

void AudioRouteHandler::convertMuteReasonToArlMuteReason(const MuteReason muteReason,
      audioroutinglib::arl_tclBitmask<arl_enMuteReason>& arlMuteReason)
{
   switch(muteReason)
   {
      case MUTE_REASON_USR_ACT:
         arlMuteReason = ARL_EN_MUTE_REASON_USR_ACT;
         break;
      case MUTE_REASON_TRACK_START:
         arlMuteReason = ARL_EN_MUTE_REASON_TRACK_START;
         break;
      case MUTE_REASON_SRC_SEL:
         arlMuteReason = ARL_EN_MUTE_REASON_SRC_SEL;
         break;
      case MUTE_REASON_DIAG:
         arlMuteReason = ARL_EN_MUTE_REASON_DIAG;
         break;
      case MUTE_REASON_OTHER:
         arlMuteReason = ARL_EN_MUTE_REASON_OTHER;
         break;
      default:
         arlMuteReason = ARL_EN_MUTE_REASON_OTHER;
         break;
   }
}

void AudioRouteHandler::convertArlSourceNumToAudioChannel(const arl_tenSource enSrcNum, AudioChannel& channel)
{
   switch(enSrcNum)
   {
      case ARL_SRC_PHONE_OUTBAND_RING:
         channel = AM_VEHICLERINGTONE;
         break;
      case ARL_SRC_PHONE:
         channel = AM_PHONEAUDIO;
         break;
      case ARL_SRC_PHONE_WAIT:
         channel = AM_WAITINGMODE;
         break;
      case ARL_SRC_PHONE_VR:
         channel = AM_VOICERECOGNITION;
         break;
      default:
         channel = AM_UNKNOWN;
         break;
   }
}

void AudioRouteHandler::convertArlSrcActivityToSrcActivity(const arl_tSrcActivity& arlSrcActivity,
      SrcActivity& srcActivity)
{
   switch(arlSrcActivity.enType)
   {
      case midw_fi_tcl_e8_SrcActivity::FI_EN_OFF:
         srcActivity = SRC_ACTIVITY_OFF;
         break;
      case midw_fi_tcl_e8_SrcActivity::FI_EN_ON:
         srcActivity = SRC_ACTIVITY_ON;
         break;
      case midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE:
         srcActivity = SRC_ACTIVITY_PAUSE;
         break;
      default:
         srcActivity = SRC_ACTIVITY_OFF;
         break;
   }
}

void AudioRouteHandler::convertArlMuteStateToMuteState(const arl_tenMuteState arlMuteState, MuteState& muteState)
{
   switch(arlMuteState)
   {
      case ARL_EN_MUTESTATE_DEMUTE:
      case ARL_EN_MUTESTATE_DEMUTE_RUNNING:
         muteState = false;
         break;
      case ARL_EN_MUTESTATE_MUTE:
         muteState = true;
         break;
      default:
         muteState = false;
         break;
   }
}

} //pmaudiomanager
