/**
 * @file ArlWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the wrapper functions of ARL requests and response methods.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef ArlWrapper_h
#define ArlWrapper_h

#include "AmIfMessageCreator.h"
#include "PmAudioManagerIfTypes.h"
#include "ArlTypes.h"

//class forward declarations
class ahl_tclBaseOneThreadApp;

namespace pmaudiomanager {

//class forward declarations
class AudioRouteHandler;

/**
 * Arl wrapper class.
 */
class ArlWrapper final
{
public:
   /**
    * Parameterized Constructor of ArlWrapper class
    *
    * @param[in] mainApplication - pointer of PM main application thread
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   ArlWrapper(ahl_tclBaseOneThreadApp* mainApplication);

   /**
    * Destructor of ArlWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   virtual ~ArlWrapper();

   /**
    * Copy constructor => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   ArlWrapper(IN const ArlWrapper& ref);

   /**
    * Assignment operator => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return reference to ArlWrapper
    */
   ArlWrapper& operator=(IN const ArlWrapper& ref);

   /**
    * This method is used to request audio routing
    *
    * @param[in] channel: audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode requestAudioRoute(IN const AudioChannel channel);

   /**
    * Callback method for audio route request
    *
    * @param[in] amResult: audio manager result
    * @param[in] channel: audio channel
    * @param[in] audioSink : audio sink devices list
    * @param[in] audioSource : audio source devices list
    * @param[out]
    * @param[in,out]
    *
    * @return bool
    */
   bool requestAudioRouteCB(IN const AmResult& amResult, IN const AudioChannel channel,
         IN const AudioDeviceList& audioSink, IN const AudioDeviceList& audioSource);

   /**
    * This method is used to release audio routing
    *
    * @param[in] channel: audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode releaseAudioRoute(IN const AudioChannel channel);

   /**
    * Callback method for audio route release request
    *
    * @param[in] amResult: audio manager result
    * @param[in] channel: audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return bool
    */
   bool releaseAudioRouteCB(IN const AmResult& amResult, IN const AudioChannel channel);

   /**
    * This method is used to set source availability
    *
    * @param[in] srcAvailability: source availability
    * @param[in] availabilityReason: availability reason
    * @param[in] channel: audio channel
    * @param[in] subSource: sub source (default = 0)
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode setSourceAvailability(IN const SrcAvailability srcAvailability,
         IN const AvailabilityReason availabilityReason, IN const AudioChannel channel,
         IN const SubSource subSource = SUB_SOURCE_DEFAULT);

   /**
    * Callback method to notify the audio source activity changes
    *
    * @param[in] channel: audio channel
    * @param[in] subSource: sub source
    * @param[in] srcActivity: source activity
    * @param[out]
    * @param[in,out]
    *
    * @return bool
    */
   bool onSourceActivityCB(IN const AudioChannel channel, IN const SubSource subSource,
         IN const SrcActivity srcActivity);

   /**
    * This method is used to request mute to Arl
    *
    * @param[in] channel: audio channel
    * @param[in] muteState: mute state
    * @param[in] muteReason: mute reason
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode requestMuteAction(IN const AudioChannel channel, IN const MuteState muteAction,
         IN const MuteReason muteReason);

   /**
    * Callback method for mute action
    *
    * @param[in] amResult: audio manager result
    * @param[in] channel: audio channel
    * @param[in] muteState: mute state
    * @param[out]
    * @param[in,out]
    *
    * @return bool
    */
   bool requestMuteActionCB(IN const AmResult& amResult, IN const AudioChannel channel, MuteState muteState);

   /**
    * This method is used to send source activity result back to Arl
    *
    * @param[in] channel: audio channel
    * @param[in] srcActivity: source activity
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode sendSourceActivityResult(IN const AudioChannel channel,IN const SrcActivity srcActivity);

protected:
   /**
    * Constructor of ArlWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   ArlWrapper();

private:

   /**
    * This method is used to post the callback message to the AM - Maincontroller
    *
    * @param[in] AmIfMessage
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return bool
    */
   bool sendCbMsg2AmMainController(IN std::shared_ptr<AmIfMessage>& amIfMessage);

   AudioRouteHandler* _audioRouteHandler; /**< pointer to AudioRouteHandler class */
};

} //pmaudiomanager

#endif // ArlWrapper_h
