/**
 * @file AmIfMessageRequest.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @briefThis file contains the definition of all AmIfMessageRequest
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the implementation for all PmAudioManager request Messages
 *
 * @ingroup PmAudioManager
 */

#include "AmIfMessageRequest.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AmIfMessageRequest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   AmIfMessageRequest::AmIfMessageRequest(IN const AmIfMsgId messageId, IN const AmSessionId amSessionId) :
         AmIfMessage(messageId, amSessionId)
   {
      ETG_TRACE_USR1(("AmIfMessageRequest: [0x%p] creating PM Audio Manager IF message object", (void*) this));
   }

   AmIfMessageRequest::AmIfMessageRequest(IN const AmIfMessageRequest& other) :
         AmIfMessage(other)
   {
      ETG_TRACE_USR1(("AmIfMessageRequest: [0x%p] copy-creating PM Audio Manager IF message object", (void*) this));
   }

   AmIfMessageRequest::~AmIfMessageRequest()
   {
      ETG_TRACE_USR1(("AmIfMessageRequest: [0x%p] destroying PM Audio Manager IF message object", (void*) this));
   }

   void swap(AmIfMessageRequest& first, AmIfMessageRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessage&>(first), static_cast<AmIfMessage&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessageRequest& AmIfMessageRequest::operator=(AmIfMessageRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessageRequest::operator==(const AmIfMessageRequest& rhs) const
   {
      return (AmIfMessage::operator ==(rhs));
   }

   void AmIfMessageRequest::traceMessage(void) const
   {
      AmIfMessage::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_PrepareAudioRouteRequest::AmIfMessage_PrepareAudioRouteRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_PREPARE_AUDIO_ROUTE, AM_SESSION_ID_DEFAULT),
         _audioChannelId(AM_UNKNOWN)
   {
      ETG_TRACE_USR1(("AmIfMessage_PrepareAudioRouteRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PrepareAudioRouteRequest::AmIfMessage_PrepareAudioRouteRequest(
         IN const AmSessionId amSessionId, IN const AudioChannel audioChannelId) :
         AmIfMessageRequest(AM_IF_MSG_ID_PREPARE_AUDIO_ROUTE, amSessionId),
         _audioChannelId(audioChannelId)
   {
      ETG_TRACE_USR1(("AmIfMessage_PrepareAudioRouteRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PrepareAudioRouteRequest::AmIfMessage_PrepareAudioRouteRequest(
         IN const AmIfMessage_PrepareAudioRouteRequest& other) : AmIfMessageRequest(other),
         _audioChannelId(other._audioChannelId)
   {
      ETG_TRACE_USR1(("AmIfMessage_PrepareAudioRouteRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PrepareAudioRouteRequest::~AmIfMessage_PrepareAudioRouteRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_PrepareAudioRouteRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_PrepareAudioRouteRequest& first, AmIfMessage_PrepareAudioRouteRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._audioChannelId, second._audioChannelId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_PrepareAudioRouteRequest& AmIfMessage_PrepareAudioRouteRequest::operator=(
         AmIfMessage_PrepareAudioRouteRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_PrepareAudioRouteRequest::operator==(const AmIfMessage_PrepareAudioRouteRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_audioChannelId == rhs._audioChannelId));
   }

   void AmIfMessage_PrepareAudioRouteRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AudioChannel = %u", (const void*) this,
            ETG_CENUM(AudioChannel, _audioChannelId)));
   }

   // *****************************************************************************************************************

   AmIfMessage_PlayAudioFileRequest::AmIfMessage_PlayAudioFileRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_PLAY_AUDIO_FILE, AM_SESSION_ID_DEFAULT),
         _filePath(""),
         _playCount(PLAY_COUNT_DEFAULT),
         _toneType(RING_TONE)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayAudioFileRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayAudioFileRequest::AmIfMessage_PlayAudioFileRequest(
         IN const AmSessionId amSessionId, IN const FilePath& filePath, IN const PlayCount playCount,
         IN const ToneType toneType) :
         AmIfMessageRequest(AM_IF_MSG_ID_PLAY_AUDIO_FILE, amSessionId),
         _filePath(filePath), _playCount(playCount), _toneType(toneType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayAudioFileRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayAudioFileRequest::AmIfMessage_PlayAudioFileRequest(
         IN const AmIfMessage_PlayAudioFileRequest& other) :
         AmIfMessageRequest(other), _filePath(other._filePath), _playCount(other._playCount),
         _toneType(other._toneType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayAudioFileRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayAudioFileRequest::~AmIfMessage_PlayAudioFileRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayAudioFileRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_PlayAudioFileRequest& first, AmIfMessage_PlayAudioFileRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._filePath, second._filePath);
      swap(first._playCount, second._playCount);
      swap(first._toneType, second._toneType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_PlayAudioFileRequest& AmIfMessage_PlayAudioFileRequest::operator=(
         AmIfMessage_PlayAudioFileRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_PlayAudioFileRequest::operator==(const AmIfMessage_PlayAudioFileRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_filePath == rhs._filePath)
            && (this->_playCount == rhs._playCount)
            && (this->_toneType == rhs._toneType));
   }

   void AmIfMessage_PlayAudioFileRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] FilePath = \"%80s\", PlayCount = %d", (const void*) this,
            _filePath.c_str(), _playCount));
   }

   // *****************************************************************************************************************

   AmIfMessage_PlayHfAudioRequest::AmIfMessage_PlayHfAudioRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_PLAY_HF_AUDIO, AM_SESSION_ID_DEFAULT),
         _samplingType(UNKNOWN)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayHfAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayHfAudioRequest::AmIfMessage_PlayHfAudioRequest(
         IN const AmSessionId amSessionId, IN const SamplingType samplingType) :
         AmIfMessageRequest(AM_IF_MSG_ID_PLAY_HF_AUDIO, amSessionId),
         _samplingType(samplingType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayHfAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayHfAudioRequest::AmIfMessage_PlayHfAudioRequest(IN const AmIfMessage_PlayHfAudioRequest& other) :
         AmIfMessageRequest(other), _samplingType(other._samplingType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayHfAudioRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlayHfAudioRequest::~AmIfMessage_PlayHfAudioRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_PlayHfAudioRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_PlayHfAudioRequest& first, AmIfMessage_PlayHfAudioRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._samplingType, second._samplingType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_PlayHfAudioRequest& AmIfMessage_PlayHfAudioRequest::operator=( AmIfMessage_PlayHfAudioRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_PlayHfAudioRequest::operator==(const AmIfMessage_PlayHfAudioRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_samplingType == rhs._samplingType));
   }

   void AmIfMessage_PlayHfAudioRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] SamplingType = %u", (const void*) this,
            ETG_CENUM(SamplingType, _samplingType)));
   }

   // *****************************************************************************************************************

   AmIfMessage_StopAudioRequest::AmIfMessage_StopAudioRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_STOP_AUDIO, AM_SESSION_ID_DEFAULT)
   {
      ETG_TRACE_USR1(("AmIfMessage_StopAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_StopAudioRequest::AmIfMessage_StopAudioRequest(IN const AmSessionId amSessionId) :
         AmIfMessageRequest(AM_IF_MSG_ID_STOP_AUDIO, amSessionId)
   {
      ETG_TRACE_USR1(("AmIfMessage_StopAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_StopAudioRequest::AmIfMessage_StopAudioRequest(IN const AmIfMessage_StopAudioRequest& other) :
         AmIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_StopAudioRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_StopAudioRequest::~AmIfMessage_StopAudioRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_StopAudioRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_StopAudioRequest& first, AmIfMessage_StopAudioRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_StopAudioRequest& AmIfMessage_StopAudioRequest::operator=( AmIfMessage_StopAudioRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_StopAudioRequest::operator==(const AmIfMessage_StopAudioRequest& rhs) const
   {
      return (AmIfMessageRequest::operator ==(rhs));
   }

   void AmIfMessage_StopAudioRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_PauseAudioRequest::AmIfMessage_PauseAudioRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_PAUSE_AUDIO, AM_SESSION_ID_DEFAULT)
   {
      ETG_TRACE_USR1(("AmIfMessage_PauseAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PauseAudioRequest::AmIfMessage_PauseAudioRequest(IN const AmSessionId amSessionId) :
         AmIfMessageRequest(AM_IF_MSG_ID_PAUSE_AUDIO, amSessionId)
   {
      ETG_TRACE_USR1(("AmIfMessage_PauseAudioRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PauseAudioRequest::AmIfMessage_PauseAudioRequest(IN const AmIfMessage_PauseAudioRequest& other) :
         AmIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_PauseAudioRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PauseAudioRequest::~AmIfMessage_PauseAudioRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_PauseAudioRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_PauseAudioRequest& first, AmIfMessage_PauseAudioRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_PauseAudioRequest& AmIfMessage_PauseAudioRequest::operator=(AmIfMessage_PauseAudioRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_PauseAudioRequest::operator==(const AmIfMessage_PauseAudioRequest& rhs) const
   {
      return (AmIfMessageRequest::operator ==(rhs));
   }

   void AmIfMessage_PauseAudioRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_SetMicMuteStateRequest::AmIfMessage_SetMicMuteStateRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_SET_MIC_MUTE_STATE, AM_SESSION_ID_DEFAULT),
         _muteState(false)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMicMuteStateRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMicMuteStateRequest::AmIfMessage_SetMicMuteStateRequest(IN const MuteState muteState) :
         AmIfMessageRequest(AM_IF_MSG_ID_SET_MIC_MUTE_STATE, AM_SESSION_ID_DEFAULT),
         _muteState(muteState)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMicMuteStateRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMicMuteStateRequest::AmIfMessage_SetMicMuteStateRequest(
         IN const AmIfMessage_SetMicMuteStateRequest& other) :
         AmIfMessageRequest(other), _muteState(other._muteState)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMicMuteStateRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMicMuteStateRequest::~AmIfMessage_SetMicMuteStateRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMicMuteStateRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_SetMicMuteStateRequest& first, AmIfMessage_SetMicMuteStateRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._muteState, second._muteState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_SetMicMuteStateRequest& AmIfMessage_SetMicMuteStateRequest::operator=(
         AmIfMessage_SetMicMuteStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_SetMicMuteStateRequest::operator==(const AmIfMessage_SetMicMuteStateRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_muteState == rhs._muteState));
   }

   void AmIfMessage_SetMicMuteStateRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] MuteState = %d", (const void*) this, _muteState));
   }

   // *****************************************************************************************************************

   AmIfMessage_GetAudioSinkVolumeRequest::AmIfMessage_GetAudioSinkVolumeRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_GET_AUDIO_SINK_VOLUME, AM_SESSION_ID_DEFAULT),
         _sinkId(1)
   {
      ETG_TRACE_USR1(("AmIfMessage_GetAudioSinkVolumeRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetAudioSinkVolumeRequest::AmIfMessage_GetAudioSinkVolumeRequest(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId) :
         AmIfMessageRequest(AM_IF_MSG_ID_GET_AUDIO_SINK_VOLUME, amSessionId),
         _sinkId(sinkId)
   {
      ETG_TRACE_USR1(("AmIfMessage_GetAudioSinkVolumeRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetAudioSinkVolumeRequest::AmIfMessage_GetAudioSinkVolumeRequest(
         IN const AmIfMessage_GetAudioSinkVolumeRequest& other) :
         AmIfMessageRequest(other), _sinkId(other._sinkId)
   {
      ETG_TRACE_USR1(("AmIfMessage_GetAudioSinkVolumeRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetAudioSinkVolumeRequest::~AmIfMessage_GetAudioSinkVolumeRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_GetAudioSinkVolumeRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_GetAudioSinkVolumeRequest& first, AmIfMessage_GetAudioSinkVolumeRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._sinkId, second._sinkId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_GetAudioSinkVolumeRequest& AmIfMessage_GetAudioSinkVolumeRequest::operator=(
		   AmIfMessage_GetAudioSinkVolumeRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_GetAudioSinkVolumeRequest::operator==(const AmIfMessage_GetAudioSinkVolumeRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_sinkId == rhs._sinkId));
   }

   void AmIfMessage_GetAudioSinkVolumeRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] SinkID = %d", (const void*) this, _sinkId));
   }

   // *****************************************************************************************************************

   AmIfMessage_SetAudioSinkVolumeRequest::AmIfMessage_SetAudioSinkVolumeRequest() :
         AmIfMessageRequest(AM_IF_MSG_ID_SET_AUDIO_SINK_VOLUME, AM_SESSION_ID_DEFAULT),
         _sinkId(0), _propertyValue({})
   {
      ETG_TRACE_USR1(("AmIfMessage_SetAudioSinkVolumeRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetAudioSinkVolumeRequest::AmIfMessage_SetAudioSinkVolumeRequest(IN const AmSessionId amSessionId, IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyValue) :
         AmIfMessageRequest(AM_IF_MSG_ID_SET_AUDIO_SINK_VOLUME, amSessionId),
         _sinkId(sinkId),
         _propertyValue(propertyValue)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetAudioSinkVolumeRequest: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetAudioSinkVolumeRequest::AmIfMessage_SetAudioSinkVolumeRequest(
         IN const AmIfMessage_SetAudioSinkVolumeRequest& other) :
         AmIfMessageRequest(other), _sinkId(other._sinkId), _propertyValue(other._propertyValue)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetAudioSinkVolumeRequest: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetAudioSinkVolumeRequest::~AmIfMessage_SetAudioSinkVolumeRequest()
   {
      ETG_TRACE_USR1(("AmIfMessage_SetAudioSinkVolumeRequest: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_SetAudioSinkVolumeRequest& first, AmIfMessage_SetAudioSinkVolumeRequest& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageRequest&>(first), static_cast<AmIfMessageRequest&>(second));
      swap(first._sinkId, second._sinkId);
      swap(first._propertyValue, second._propertyValue);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_SetAudioSinkVolumeRequest& AmIfMessage_SetAudioSinkVolumeRequest::operator=(
         AmIfMessage_SetAudioSinkVolumeRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_SetAudioSinkVolumeRequest::operator==(const AmIfMessage_SetAudioSinkVolumeRequest& rhs) const
   {
      return ((AmIfMessageRequest::operator ==(rhs))
            && (this->_sinkId == rhs._sinkId)
            && (this->_propertyValue == rhs._propertyValue));
   }

   void AmIfMessage_SetAudioSinkVolumeRequest::traceMessage(void) const
   {
      AmIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] SinkID = %d", (const void*) this, _sinkId));
   }

   // *****************************************************************************************************************

}// namespace pmaudiomanager
