/*
 * Lock.cpp
 *
 *  Created on: 28.01.2015
 *      Author: Thömel
 */

#include "FwUtils.h"

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Lock.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#endif
#endif
#include "TraceDefinitions.h"
#include "FunctionTracer.h"

#include "Lock.h"

#include "Utils.h"

Lock::Lock()
{
    /* init the lock */
    pthread_mutexattr_t mta;
    FW_FATAL_ASSERT(pthread_mutexattr_init(&mta) == 0);
    FW_FATAL_ASSERT(pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE) == 0);
    FW_FATAL_ASSERT(pthread_mutex_init(&mutex, &mta) == 0);

    /* set some members */
    mToValue = 120; // default time out value
    isLocked = 0;  // lock not locked
}

Lock::Lock(const bool reentrant, const unsigned int sec)
{
    /* init the lock */
    pthread_mutexattr_t mta;
    FW_FATAL_ASSERT(pthread_mutexattr_init(&mta) == 0);
    if(reentrant)
    {
        FW_FATAL_ASSERT(pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE) == 0);
    }
    FW_FATAL_ASSERT(pthread_mutex_init(&mutex, &mta) == 0);

    /* set some members */
    mToValue = sec; // time out value
    isLocked = 0;  // lock not locked
}

int Lock::lock()
{
    struct timespec abs_time;
    int err;

relock:
    clock_gettime(CLOCK_REALTIME , &abs_time);
    abs_time.tv_sec += mToValue; // try it

    while(1) {
        if (!mToValue) {
            err = pthread_mutex_lock(&mutex);
        } else {
            err = pthread_mutex_timedlock(&mutex, &abs_time);
        }

        /* error? */
        switch(err) {
        case EINVAL:
            /*
             * The mutex was created with the protocol attribute having the value PTHREAD_PRIO_PROTECT
             * and the calling thread's priority is higher than the mutex's current priority ceiling.
             */
            break;
        case EBUSY:
            /*
             * The mutex could not be acquired because it was already locked.
             * So this call to lock() was successful
             */
            return 0;
        case EDEADLK:
            /*
             * The current thread already owns the mutex.
             * So this call to lock() was successful (in case for recursive lock enabled)
             */
            return 0;
        case EAGAIN:
            /*
             * The mutex could not be acquired because the maximum number of recursive locks for
             * mutex has been exceeded.
             * retry after a short sleep it until this maximum number allows a lock.
             */
            usleep(16L*1000L);
            goto relock;
        case ETIMEDOUT:
            /*
             * The mutex could not be locked before the specified timeout expired.
             * End the loop here to prevent endless blocking - but it is severe!
             */
            break;
        case EPERM:
            /*
             * The current thread does not own the mutex.
             * Fatal error, cannot handle with this case - software design error.
             */
            break;
        case EINTR:
            /*
             * this should not happen but to be secure...
             * In former times a signal can cause a return of a lock call. But the lock was not acquired.
             * Retry to lock it
             */
            continue;
        }

        /*
         * unknown or no error:
         * break the lock loop and go to the timeout value check.
         */
        break;
    }

    if ((EINVAL == err) || (EPERM == err))
    {
       ETG_TRACE_FATAL(("lock: mutex could not be acquired (err = %d)", err));
       ETG_TRACE_ERRMEM(("lock: mutex could not be acquired (err = %d)", err));

       return -1;
    }

    /*
     * in the case of timeout or unexpected error: re-check the current system time against the calculated absolute timeout value.
     * Was the system clock adjusted?: If yes: re-start the lock loop with a new absolute timeout value.
     */
    if (err != 0) {

        /* if lock tried with timeout */
        if (mToValue) {

            /* check if the timeout is plausible */
            struct timespec absTimeAfterTimeout;
            clock_gettime(CLOCK_REALTIME , &absTimeAfterTimeout);

            /* check if the calculated timeout time matches with the current real time */
            if (::fw::calcAbsoluteDifference(abs_time.tv_sec, absTimeAfterTimeout.tv_sec) > 1) {

                /* timeout was not plausible, maybe system clock was adjusted */
                goto relock;
            }

            // timeout was plausible
            ETG_TRACE_FATAL(("lock: mutex could not be acquired within %d seconds (err = %d), mutex got released to prevent from endless blocking", err, mToValue));
            ETG_TRACE_ERRMEM(("lock: mutex could not be acquired within %d seconds (err = %d), mutex got released to prevent from endless blocking", err, mToValue));
        }
        else
        {
           ETG_TRACE_FATAL(("lock: mutex could not be acquired (err = %d)", err));
           ETG_TRACE_ERRMEM(("lock: mutex could not be acquired (err = %d)", err));
        }

        /* return error */
        return -1;
    } else {
        isLocked = 1;
    }
    return 0;
}

int Lock::tryLock()
{
   /*
    * https://linux.die.net/man/3/pthread_mutex_lock:
    * The pthread_mutex_trylock() function shall be equivalent to pthread_mutex_lock(),
    * except that if the mutex object referenced by mutex is currently locked (by any thread, including the current thread),
    * the call shall return immediately.
    * If the mutex type is PTHREAD_MUTEX_RECURSIVE and the mutex is currently owned by the calling thread,
    * the mutex lock count shall be incremented by one and the pthread_mutex_trylock() function shall immediately return success.
    */

   int result;
   bool error(false);

   while((result = pthread_mutex_trylock(&mutex)) != 0)
   {
      /*
       * The pthread_mutex_trylock() function shall return zero if a lock on the mutex object referenced by mutex is acquired. Otherwise, an error number is returned to indicate the error.
       * The pthread_mutex_trylock() functions shall fail if:
       * EINVAL: The mutex was created with the protocol attribute having the value PTHREAD_PRIO_PROTECT and the calling thread's priority is higher than the mutex's current priority ceiling.
       * EBUSY: The mutex could not be acquired because it was already locked.
       * EINVAL: The value specified by mutex does not refer to an initialized mutex object.
       * EAGAIN: The mutex could not be acquired because the maximum number of recursive locks for mutex has been exceeded.
       */

      switch(result)
      {
         case EINVAL:
            ETG_TRACE_FATAL(("tryLock: mutex could not be acquired (err = %d)", result));
            error = true;
            break;
         case EBUSY:
            // type is not PTHREAD_MUTEX_RECURSIVE: already locked (valid for any thread, including the current thread) => try again (current thread: dead lock situation shall be solved outside or use type PTHREAD_MUTEX_RECURSIVE)
            // type is PTHREAD_MUTEX_RECURSIVE: already locked (valid for non calling thread) => try again
            break;
         case EAGAIN:
            // retry after a short sleep until this maximum number allows a lock
            usleep(16L * 1000L);
            break;
         default:
            ETG_TRACE_FATAL(("tryLock: mutex could not be acquired (err = %d)", result));
            error = true;
            break;
      }

      if(true == error)
      {
         return -1;
      }

      // TODO: sleep
      usleep(16L * 1000L);
   }

   // mutex was acquired successfully
   isLocked = 1;
   return 0;
}

void Lock::unlock()
{
    pthread_mutex_unlock(&mutex);
    isLocked = 0;
}

Lock::~Lock()
{
    pthread_mutex_unlock(&mutex);
    pthread_mutex_destroy(&mutex);
    isLocked = 0;
}

int Lock::setNotReantrant()
{
    pthread_mutexattr_t mta;
    pthread_mutexattr_init(&mta);
    return pthread_mutex_init(&mutex, &mta);
}

void Lock::setTimeout(const unsigned int sec)
{
    mToValue = sec;
}

LockForever::LockForever() : Lock(true, 0U)
{
}

LockForever::~LockForever()
{
}

int LockForever::setNotReantrant()
{
   ETG_TRACE_FATAL(("setNotReantrant: mutex is not allowed to be set non-reentrant"));
   ETG_TRACE_ERRMEM(("setNotReantrant: mutex is not allowed to be set non-reentrant"));
   FW_NORMAL_ASSERT_ALWAYS();

   return -1;
}

void LockForever::setTimeout(const unsigned int sec)
{
   (void)(sec);
   ETG_TRACE_FATAL(("setTimeout: not allowed to set a timeout for mutex"));
   ETG_TRACE_ERRMEM(("setTimeout: not allowed to set a timeout for mutex"));
   FW_NORMAL_ASSERT_ALWAYS();
}

LockForeverAndNonReentrant::LockForeverAndNonReentrant() : Lock(false, 0U)
{
}

LockForeverAndNonReentrant::~LockForeverAndNonReentrant()
{
}

int LockForeverAndNonReentrant::setNotReantrant()
{
   ETG_TRACE_FATAL(("setNotReantrant: mutex is already set to be non-reentrant"));
   ETG_TRACE_ERRMEM(("setNotReantrant: mutex is already set to be non-reentrant"));
   FW_NORMAL_ASSERT_ALWAYS();

   return -1;
}

void LockForeverAndNonReentrant::setTimeout(const unsigned int sec)
{
   (void)(sec);
   ETG_TRACE_FATAL(("setTimeout: not allowed to set a timeout for mutex"));
   ETG_TRACE_ERRMEM(("setTimeout: not allowed to set a timeout for mutex"));
   FW_NORMAL_ASSERT_ALWAYS();
}
