/**
 * @file FwSingleThread.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Single thread.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Single thread implementation.
 */

#include "FwSingleThread.h"
#include "FwISingleThreadFunction.h"
#include "TraceDefinitions.h"
#include "FwAssert.h"
#include "FwTrace.h"

#include <cstring>
#include <cerrno>
#include <sys/prctl.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwSingleThread.cpp.trc.h"
#endif
#endif

namespace fw {

SingleThread::SingleThread() :
_threadIf(0),
_handle(0),
_name(),
_context(0),
_semaphore()
{
   _semaphore.create();
}

SingleThread::~SingleThread()
{
   _threadIf = 0;
   _context = 0;
}

void SingleThread::start(ISingleThreadFunction* threadFunctionIf, const ::std::string& threadName, void* arguments)
{
   FW_IF_NULL_PTR_RETURN(threadFunctionIf);

   // store given data
   _threadIf = threadFunctionIf;
   if(threadName.length() < 16)
   {
      _name = threadName;
   }
   else
   {
      _name = threadName.substr(threadName.length() - 15);
   }
   _context = arguments;

   int result;

   // create thread
   for(int i = 0; i < 3; i++)
   {
      result = pthread_create(&_handle, 0 /* no attributes */, &internalThread, (void*)this);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_create.html:
            The pthread_create() function shall fail if:

            [EAGAIN]
              The system lacked the necessary resources to create another thread, or the system-imposed limit on the total number of threads in a process {PTHREAD_THREADS_MAX} would be exceeded.
            [EPERM]
              The caller does not have appropriate permission to set the required scheduling parameters or scheduling policy.
              => should never happen

            The pthread_create() function may fail if:

            [EINVAL]
              The attributes specified by attr are invalid.
              => should never happen because no attribute setting is given

            The pthread_create() function shall not return an error code of [EINTR].
          *
          */

         if(EAGAIN == result)
         {
            // try again
         }
         else
         {
            ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
            FW_NORMAL_ASSERT_ALWAYS();
            return;
         }
      }
      else
      {
         // success
         break;
      }
   }

   if(0 != result)
   {
      ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }
}

void SingleThread::stop(void)
{
   FW_IF_NULL_PTR_RETURN(_threadIf);

   ETG_TRACE_ERR((" internalThread(): thread %s to be stopped", _name.c_str()));

   _threadIf->setTerminate(_context);

   _semaphore.wait();
}

void* SingleThread::internalThread(void* context)
{
   FW_IF_NULL_PTR_RETURN_NULL(context);

   SingleThread* thread = (SingleThread*)context;

   // set thread name
   (void)prctl(PR_SET_NAME, thread->_name.c_str(), 0, 0, 0);

   FW_IF_NULL_PTR_RETURN_NULL(thread->_threadIf);

   ETG_TRACE_ERR((" internalThread(): thread %s started", thread->_name.c_str()));

   thread->_threadIf->threadFunction(thread->_context);

   ETG_TRACE_ERR((" internalThread(): thread %s closed", thread->_name.c_str()));

   thread->_semaphore.post();

   pthread_exit(0);

   return 0;
}

} //fw
