/**
 * @file FwMemoryChecker.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Memory checker.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Memory checker implementation.
 */

#include "FwMemoryChecker.h"
#include "TraceDefinitions.h"
#include "FwTrace.h"

#include <cstring>
#include <cstdlib>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwMemoryChecker.cpp.trc.h"
#endif
#endif

namespace fw {

MemoryChecker::MemoryChecker() :
_checkValue64(63),
_checkValue128(127),
_checkValue256(255),
_checkValue512(511),
_counter(0),
_newCounter(0),
_delCounter(0),
_allocatedMemory(0),
_lock(),
_ptrList(),
_memList()
{
}

MemoryChecker::~MemoryChecker()
{
}

MemoryChecker& MemoryChecker::getInstance(void)
{
   static MemoryChecker factory;
   return factory;
}

void MemoryChecker::memoryCheckNewObj(void* pointer, const char* fileName, const unsigned int lineNumber)
{
   memoryCheckNewObj(pointer, 0, fileName, lineNumber);
}

void MemoryChecker::memoryCheckNewObj(void* pointer, const size_t nmbBytes, const char* fileName, const unsigned int lineNumber)
{
   if(0 == pointer)
   {
      return;
   }

   _lock.lock();

   ++_newCounter;

   const char* tmp = "";

   if(fileName)
   {
      tmp = strrchr(fileName, '/');
      if(tmp)
      {
         tmp = tmp + 1;
      }
      else
      {
         tmp = "";
      }
   }

   if(_ptrList.end() != _ptrList.find((uintptr_t)pointer))
   {
      ETG_TRACE_ERR((" memoryCheckNewObj(): 0x%08x [%u, %s] already in list", (uintptr_t)pointer, lineNumber, tmp));
   }
   else
   {
      _ptrList[(uintptr_t)pointer] = 0;
      if(0 != nmbBytes)
      {
         _memList[(uintptr_t)pointer] = nmbBytes;
         _allocatedMemory += nmbBytes;
      }
   }

   ETG_TRACE_ERR((" memoryCheckNewObj(): 0x%08x [%u, %s]", (uintptr_t)pointer, lineNumber, tmp));

   _lock.unlock();
}

void MemoryChecker::memoryCheckDelObj(void* pointer, const char* fileName, const unsigned int lineNumber)
{
   if(0 == pointer)
   {
      return;
   }

   _lock.lock();

   ++_delCounter;

   const char* tmp = "";

   if(fileName)
   {
      tmp = strrchr(fileName, '/');
      if(tmp)
      {
         tmp = tmp + 1;
      }
      else
      {
         tmp = "";
      }
   }

   ::std::map< uintptr_t, unsigned int >::iterator it = _ptrList.find((uintptr_t)pointer);

   if(_ptrList.end() != it)
   {
      _ptrList.erase(it);

      ETG_TRACE_ERR((" memoryCheckDelObj(): 0x%08x [%u, %s] found", (uintptr_t)pointer, lineNumber, tmp));

      ::std::map< uintptr_t, size_t >::iterator it2 = _memList.find((uintptr_t)pointer);

      if(_memList.end() != it2)
      {
         if(_allocatedMemory >= it2->second)
         {
            _allocatedMemory -= it2->second;
         }
         _memList.erase(it2);
      }
   }
   else
   {
      ETG_TRACE_ERR((" memoryCheckDelObj(): 0x%08x [%u, %s] unknown", (uintptr_t)pointer, lineNumber, tmp));
   }

   memoryCheckList();

   _lock.unlock();
}

void MemoryChecker::memoryCheckList(void)
{
   ++_counter;

   if((0 == (_checkValue64 & _counter)) || (_newCounter == _delCounter))
   {
      ETG_TRACE_ERR((" memoryCheckList()  : new=%u del=%u mem=%u", _newCounter, _delCounter, _allocatedMemory));

      for(::std::map< uintptr_t, unsigned int >::iterator it = _ptrList.begin(); it != _ptrList.end(); ++it)
      {
         ++it->second;

         ETG_TRACE_ERR((" memoryCheckList()  : 0x%08x [count=%u]", it->first, it->second));
      }
   }
}

} //fw

extern "C"
{

void* mem_check_malloc(size_t size)
{
   void* tmp = malloc(size);
   ETG_TRACE_ERR((" mem_check_malloc (): 0x%08x [%u]", (uintptr_t)tmp, size));
   FW_MEMORY_CHECK_NEW_OBJ_SIZE(tmp, size);
   return tmp;
}

void mem_check_free(void* ptr)
{
   ETG_TRACE_ERR((" mem_check_free   (): 0x%08x", (uintptr_t)ptr));
   FW_MEMORY_CHECK_DEL_OBJ(ptr);
   free(ptr);
}

void* mem_check_realloc(void* ptr, size_t size)
{
   void* tmp = realloc(ptr, size);
   ETG_TRACE_ERR((" mem_check_realloc(): 0x%08x => 0x%08x [%u]", (uintptr_t)ptr, (uintptr_t)tmp, size));
   FW_MEMORY_CHECK_DEL_OBJ(ptr);
   FW_MEMORY_CHECK_NEW_OBJ_SIZE(tmp, size);
   return tmp;
}

} // extern "C"
