/**
 * @file FwFormattedDataPrint.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Formatted data print.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Formatted data print.
 */

#include "FwFormattedDataPrint.h"
#include "FwAssert.h"
#define VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
// #undef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
#include <boost/config/warning_disable.hpp>
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wconversion"
#pragma GCC diagnostic ignored "-Wunused-local-typedefs"
#pragma GCC diagnostic ignored "-Wtype-limits"
#include <boost/spirit/include/karma.hpp>
#pragma GCC diagnostic pop
#include <cfloat>
#else
// #define VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
#undef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
#include <cstdio>
#include <cfloat>
#else
#include <cfloat>
#include <iostream>
#include <strstream>
#include <iomanip>
#endif
#endif

namespace fw {

#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
// http://www.boost.org/doc/libs/1_63_0/libs/spirit/doc/html/spirit/karma.html
// http://www.boost.org/doc/libs/1_63_0/libs/spirit/workbench/karma/int_generator.cpp
// http://www.boost.org/doc/libs/1_63_0/libs/spirit/doc/html/spirit/karma/performance_measurements/numeric_performance/int_performance.html
// http://www.boost.org/doc/libs/1_63_0/libs/spirit/doc/html/spirit/karma/performance_measurements/numeric_performance/double_performance.html
// http://www.boost.org/doc/libs/1_63_0/libs/spirit/doc/html/spirit/karma/performance_measurements/numeric_performance/format_performance.html

template <typename T>
struct float_scientific_policy : boost::spirit::karma::real_policies<T>
{
   static unsigned int precision(T) { return FLT_DIG; }
   // we want the numbers always to be in scientific format
   static int floatfield(T) { return ::boost::spirit::karma::real_policies<T>::fmtflags::scientific; }
};

typedef ::boost::spirit::karma::real_generator< float, float_scientific_policy<float> > float_scientific_type;
static float_scientific_type const float_scientific = float_scientific_type();

template <typename T>
struct float_fixed_policy : boost::spirit::karma::real_policies<T>
{
   static unsigned int precision(T) { return FLT_DIG; }
   // we want the numbers always to be in fixed format
   static int floatfield(T) { return ::boost::spirit::karma::real_policies<T>::fmtflags::fixed; }
};

typedef ::boost::spirit::karma::real_generator< float, float_fixed_policy<float> > float_fixed_type;
static float_fixed_type const float_fixed = float_fixed_type();

template <typename T>
struct double_scientific_policy : boost::spirit::karma::real_policies<T>
{
   static unsigned int precision(T) { return DBL_DIG; }
   // we want the numbers always to be in scientific format
   static int floatfield(T) { return ::boost::spirit::karma::real_policies<T>::fmtflags::scientific; }
};

typedef ::boost::spirit::karma::real_generator< double, double_scientific_policy<double> > double_scientific_type;
static double_scientific_type const double_scientific = double_scientific_type();

template <typename T>
struct double_fixed_policy : boost::spirit::karma::real_policies<T>
{
   static unsigned int precision(T) { return DBL_DIG; }
   // we want the numbers always to be in fixed format
   static int floatfield(T) { return ::boost::spirit::karma::real_policies<T>::fmtflags::fixed; }
};

typedef ::boost::spirit::karma::real_generator< double, double_fixed_policy<double> > double_fixed_type;
static double_fixed_type const double_fixed = double_fixed_type();
#endif

FormattedOutputU8::FormattedOutputU8(const uint8_t data, const bool printAsHex /*= false*/)
{
   if(true == printAsHex)
   {
      // 0x00 .. 0xFF
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      _buffer[0] = '0';
      _buffer[1] = 'x';
      char* ptr = &_buffer[2];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(2, '0')[::boost::spirit::karma::hex],     // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "0x%02hhX", data);
      if(4 != nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      // next line: without casting value is not printed to buffer
      outStream << "0x" << ::std::hex << ::std::setw(2) << ::std::setfill('0') << (unsigned int)data << ::std::dec << ::std::ends;
#endif
#endif
   }
   else
   {
      // 0 .. 255
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::uint_,                                                // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%hhu", data);
      if((0 < nmb) && (nmb < 4))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      // next line: without casting value is not printed to buffer
      outStream << ::std::dec << (unsigned int)data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputU32::FormattedOutputU32(const uint32_t data, const bool printAsHex /*= false*/)
{
   if(true == printAsHex)
   {
      // 0x00000000 .. 0xFFFFFFFF
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      _buffer[0] = '0';
      _buffer[1] = 'x';
      char* ptr = &_buffer[2];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(8, '0')[::boost::spirit::karma::hex],     // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "0x%08lX", data);
      if(10 != nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << "0x" << ::std::hex << ::std::setw(8) << ::std::setfill('0') << data << ::std::dec << ::std::ends;
#endif
#endif
   }
   else
   {
      // 0 .. 4294967296
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::uint_,                                                // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%lu", data);
      if((0 < nmb) && (nmb < 11))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << ::std::dec << data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputS64::FormattedOutputS64(const int64_t data, const bool printAsHex /*= false*/)
{
   if(true == printAsHex)
   {
      uint64_t tmp = (uint64_t)data;
      // 0x0000000000000000 .. 0xFFFFFFFFFFFFFFFF
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      _buffer[0] = '0';
      _buffer[1] = 'x';
      char* ptr = &_buffer[2];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(16, '0')[::boost::spirit::karma::hex],    // the generator
               tmp                                                                           // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "0x%016llX", tmp);
      if(18 != nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << "0x" << ::std::hex << ::std::setw(16) << ::std::setfill('0') << tmp << ::std::dec << ::std::ends;
#endif
#endif
   }
   else
   {
      // -9223372036854775807 .. 0 .. 9223372036854775807
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::long_long,                                            // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%lld", data);
      if((0 < nmb) && (nmb < 21))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << ::std::dec << data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputU64::FormattedOutputU64(const uint64_t data, const bool printAsHex /*= false*/)
{
   if(true == printAsHex)
   {
      // 0x0000000000000000 .. 0xFFFFFFFFFFFFFFFF
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      _buffer[0] = '0';
      _buffer[1] = 'x';
      char* ptr = &_buffer[2];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(16, '0')[::boost::spirit::karma::hex],    // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "0x%016llX", data);
      if(18 != nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << "0x" << ::std::hex << ::std::setw(16) << ::std::setfill('0') << data << ::std::dec << ::std::ends;
#endif
#endif
   }
   else
   {
      // 0 .. 18446744073709551615
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::ulong_long,                                           // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%llu", data);
      if((0 < nmb) && (nmb < 21))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << ::std::dec << data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputPtr::FormattedOutputPtr(const uintptr_t data, const bool printAsHex /*= true*/)
{
   if(true == printAsHex)
   {
      // 0x0000000000000000 .. 0xFFFFFFFFFFFFFFFF
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      _buffer[0] = '0';
      _buffer[1] = 'x';
      char* ptr = &_buffer[2];
#if __WORDSIZE == 64
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(16, '0')[::boost::spirit::karma::hex],    // the generator
               data                                                                          // the data to output
               );
#else
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::right_align(8, '0')[::boost::spirit::karma::hex],     // the generator
               data                                                                          // the data to output
               );
#endif
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "0x%016llX", data);
      if(18 != nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << "0x" << ::std::hex << ::std::setw((sizeof(uintptr_t) << 1)) << ::std::setfill('0') << data << ::std::dec << ::std::ends;
#endif
#endif
   }
   else
   {
      // 0 .. 18446744073709551615
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::ulong_long,                                           // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%llu", data);
      if((0 < nmb) && (nmb < 21))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << ::std::dec << data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputPid::FormattedOutputPid(const pid_t data)
{
   {
      // 0 .. 18446744073709551615
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               ::boost::spirit::karma::long_long,                                            // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%lld", data);
      if((0 < nmb) && (nmb < 21))
      {
         // OK
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream << ::std::dec << data << ::std::ends;
#endif
#endif
   }
}

FormattedOutputFloat::FormattedOutputFloat(const float data, const bool printAsScientific /*= false*/)
{
   if(true == printAsScientific)
   {
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               float_scientific,                                                             // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%.*e", FLT_DIG, data);
      if(0 > nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream.precision(FLT_DIG);
      outStream << ::std::scientific << data << ::std::ends;
      // outStream.unsetf(::std::ios::floatfield);
#endif
#endif
   }
   else
   {
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               float_fixed,                                                                  // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%.*f", FLT_DIG, data);
      if(0 > nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream.precision(FLT_DIG);
      // outStream.setf(::std::_S_fixed, ::std::_S_floatfield);
      // outStream.unsetf(::std::_S_floatfield);
      outStream << ::std::fixed << data << ::std::ends;
      // outStream << data << ::std::ends;
      // outStream.unsetf(::std::ios::floatfield);
#endif
#endif
   }
}

FormattedOutputDouble::FormattedOutputDouble(const double data, const bool printAsScientific /*= false*/)
{
   if(true == printAsScientific)
   {
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               double_scientific,                                                            // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%.*e", DBL_DIG, data);
      if(0 > nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream.precision(DBL_DIG);
      outStream << ::std::scientific << data << ::std::ends;
      // outStream.unsetf(::std::ios::floatfield);
#endif
#endif
   }
   else
   {
#ifdef VARIANT_S_FTR_ENABLE_FW_BOOST_KARMA
      char* ptr = &_buffer[0];
      const bool result = ::boost::spirit::karma::generate(
               ptr,                                                                          // destination: output iterator
               double_fixed,                                                                 // the generator
               data                                                                          // the data to output
               );
      FW_NORMAL_ASSERT(true == result);
      *ptr = '\0';
#else
#ifdef VARIANT_S_FTR_ENABLE_FW_OLD_FORMATTED_DATA_PRINT
      int nmb = snprintf(_buffer, sizeof(_buffer), "%.*f", DBL_DIG, data);
      if(0 > nmb)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#else
      ::std::ostrstream outStream(_buffer, sizeof(_buffer));
      outStream.precision(DBL_DIG);
      outStream << ::std::fixed << data << ::std::ends;
      // outStream.unsetf(::std::ios::floatfield);
#endif
#endif
   }
}

} //fw
