/**
 * @file FwEventMaster.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Event master.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Event master implementation.
 */

#include "FwEventMaster.h"
#include "FwIEventHandler.h"
#include "FwAssert.h"
#include "TraceDefinitions.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwEventMaster.cpp.trc.h"
#endif
#endif

namespace fw {

EventMaster::EventMaster() :
_created(false),
_thread(),
_lock(),
_eventLists(),
_terminate(false),
_eventSem(),
_defaultThreadName("CONN_EV_MASTER")
{
}

EventMaster::~EventMaster()
{
}

void EventMaster::create(const ::std::string& threadName, const unsigned int numberPriorities /*= 1*/)
{
   _lock.lock();

   if(false == _created)
   {
      unsigned int nmb = 1;
      if(numberPriorities > nmb)
      {
         nmb = numberPriorities;
      }

      _eventLists.clear();
      _eventLists.reserve(nmb);
      _eventLists.resize(nmb);

      _terminate = false;

      _eventSem.create();

      if(true == threadName.empty())
      {
         _thread.start(this, _defaultThreadName, 0);
      }
      else
      {
         _thread.start(this, threadName, 0);
      }

      _created = true;
   }

   _lock.unlock();
}

void EventMaster::destroy(void)
{
   _lock.lock();

   if(true == _created)
   {
      _eventLists.clear();

      _terminate = true;

      _eventSem.post();

      // TODO: destroy sem?
      // TODO: wait for end of thread?

      _created = false;
   }

   _lock.unlock();
}

void EventMaster::push(IEventHandler* eventHandler, const unsigned int priority)
{
   FW_IF_NULL_PTR_RETURN(eventHandler);

   _lock.lock();

   if(true == _created)
   {
      if(_eventLists.size() > priority)
      {
         _eventLists[priority].push(eventHandler);

         _eventSem.post();
      }
      else
      {
         // priority value is out of range
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   else
   {
      // create() method was not called
      FW_NORMAL_ASSERT_ALWAYS();
   }

   _lock.unlock();
}

void EventMaster::threadFunction(void* arguments)
{
   (void)arguments;

   IEventHandler* handler;
   size_t i;
   bool found;

   while(false == _terminate)
   {
      handler = 0;
      i = 0; // 0 indicates highest priority
      found = false;

      _lock.lock();

      const size_t nmbEventQueues(_eventLists.size());

      while((false == found) && (nmbEventQueues > i))
      {
         if(0 < _eventLists[i].size())
         {
            // get and remove handler
            handler = _eventLists[i].front();
            _eventLists[i].pop();
            found = true;
         }
         else
         {
            ++i;
         }
      }

      _lock.unlock();

      if(0 != handler)
      {
         handler->processEvent();
      }

      _eventSem.wait();
   }
}

} //fw
