/*
 * MessageQueue.h
 *
 *  Created on: 13.02.2015
 *      Author: Matthias Thömel
 *
 *  Method symmetry:
 *
 *  Send a message:
 *  - msgQueue.CreateMessageBuffer
 *  - copy data into the buffer
 *  - msgQueue.Push
 *
 *  Receive a message:
 *  - msgQueue.WaitForMessage
 *  - copy data from the buffer
 *  - msgQueue.DropMessage
 *
 *  Peek for a message:
 *  - msgQueue.Peek
 *  - copy data from buffer
 *
 *  Remove a message by message number:
 *  - msgQueue.Remove (calls DropMessage)
 *
 *  Flush all messages in queue:
 *  - msgQueue.Flush (calls DropMessage for all messages)
 */

#ifndef MESSAGEQUEUE_H_
#define MESSAGEQUEUE_H_

#include <map>
#include <string>
#include <vector>
#include <pthread.h>

#include <semaphore.h>

#include "BaseTypes.h"
#include "Lock.h"

using namespace std;

#define OLD_IMPL 0

class MessageQueue
{
public:
    MessageQueue(const char *name);
    virtual ~MessageQueue(void);

    /*
     * pushes a new message into the queue respecting its priority.
     * Use CreateMessageBuffer to get a send buffer
     */
    tResult Push(const void* message, size_t msgSize, int priority);

    /*
     * ATTENTION: ONLY FOR TEST PURPOSES
     * pops out highest priority element from the queue
     * Note : use DropMessage to release message buffer
     * Note: the counting semaphore is not affected, use WaitForMessage instead.
     */
    tResult Pop(void** message , size_t *msgSize);

    /*
     * pops out highest priority element from the queue , waits if the queue is empty
     * Note : use DropMessage to release message buffer
     */
    void* WaitForMessage (size_t *msgSize,unsigned int time_out_in_ms);

    /*
     * Methods to create and drop message buffer.
     * ATTENTION: Must be used to create the buffer before sending and to release buffer after receive
     */
    void *CreateMessageBuffer (size_t size);
    tResult DropMessage (void* message );

    /*
     * returns the message count in the queue
     */
    uint64_t GetCurMessagesCount();

    /*
     * removes all the message from the queue and deallocates memory used by the messages
     */
    tResult Flush();

    /*
     * additional methods to enable the SMF to delete specific message from the queue
     */
    tResult DoLock();
    tResult UnLock();
    void * Peek(const int iMessageNumber, size_t *msgSize);
    tResult Remove(const int iMessageNumber);
    const char *GetName() {return mName;};

private:
    MessageQueue(const MessageQueue& ref);
    MessageQueue& operator=(const MessageQueue& ref);

#if OLD_IMPL
    map<int ,vector<void*> > mMessages; /**< pair of key "priority" to  value "vector container as queue holding messages with this priority"*/
#else // OLD_IMPL

    enum {mIntegralSize = 128}; // number of messages for a alloc/realloc
    enum {mMaxPriority = 256};
    typedef struct {
        const void *message;
        size_t size;
    } tQueueElement;
    typedef struct QueueArrayElement {
        tQueueElement *head; // points to first used queue element(if head == tail: queue is empty)
        tQueueElement *tail; // points to next free queue element
        tQueueElement *base;
        tQueueElement *end;
        size_t size;
    } tQueueArrayElement;
    tQueueArrayElement mQueueArray[mMaxPriority];

#endif // OLD_IMPL

    const char *mName;                  /**< name of the message queue.*/
    Lock mAccessLock;
    sem_t mCountingSemaphore;

    tResult ReleaseMessageWait(void);
    tResult StartMessageWait(unsigned int milliSeconds);
};

#endif /* MESSAGEQUEUE_H_ */
