/**
 * @file FwStringUtils.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief String utilities.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of string utilities.
 */

#ifndef _FW_STRING_UTILS_H_
#define _FW_STRING_UTILS_H_

#include <string>
#include <vector>
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled

namespace fw {

/**
 * Convert byte to hex string.
 *
 * @param[out] hexString: hex string (converted data); size of given buffer must be at least 3 bytes
 * @param[in] binaryValue: binary data (byte)
 * @param[in] lowerCase: flag for converting to lower case characters
 */
void convertByte2HexString(char* hexString, const unsigned char binaryValue, const bool lowerCase = true);

/**
 * Convert hex string to byte.
 *
 * @param[in] hexString: hex string; size of given buffer must be at least 3 bytes
 *
 * @return = byte (converted data)
 */
unsigned char convertHexString2Byte(const char* hexString);

/**
 * Convert binary data to hex string.
 *
 * @param[out] stringValue: hex string (converted data)
 * @param[in] binaryValue: binary data
 * @param[in] lowerCase: flag for converting to lower case characters
 */
void convertBinary2HexString(::std::string& stringValue, const ::std::vector< unsigned char >& binaryValue, const bool lowerCase = true);

/**
 * Convert hex string (ASCII) to binary data.
 *
 * @param[out] binaryValue: binary data (converted data)
 * @param[in] stringValue: hex string
 *
 * @return = true: conversion was successful,
 * @return = false: conversion failed
 */
bool convertHexString2Binary(::std::vector< unsigned char >& binaryValue, const ::std::string& stringValue);

/**
 * Convert ASCII string to lower case characters.
 *
 * @param[in,out] value: string
 */
void convertString2LowerCase(::std::string& value);

/**
 * Convert ASCII string to upper case characters.
 *
 * @param[in,out] value: string
 */
void convertString2UpperCase(::std::string& value);

/**
 * Check given string for NULL and return valid string (not a NULL pointer).
 *
 * @param[in] input: input string
 *
 * @return = valid string
 */
inline const char* stringIsNeverNull(const char* input) { if(0 != input) { return input; } else { return ""; } }

/**
 * Verify given string that content is a valid UTF-8 string. Invalid UTF-8 byte sequences are replaced.
 *
 * @param[in,out] value: string
 * @param[in] replacement: replacement character
 *
 * @return = false: no byte/character replaced (valid UTF-8 string was given),
 * @return = true: at least one byte/character replaced (invalid UTF-8 string was given)
 */
bool makeValidUtf8String(::std::string& value, const char replacement = '?');

/**
 * Convert hex string to BCD code. Only characters '0' .. '9' are allowed as input.
 *
 * @param[out] bcdCode: BCD code
 * @param[in] stringValue: hex string
 * @param[in] requiredLength: required length of string to be converted
 *
 * @return = true: conversion was successful,
 * @return = false: conversion failed
 */
bool convertHexString2BcdCode(uint64_t& bcdCode, const ::std::string& stringValue, const size_t requiredLength);

} //fw

#endif //_FW_STRING_UTILS_H_
