/**
 * @file FwBluetoothStringUtils.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief String utilities.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of Bluetooth specific string utilities.
 */

#ifndef _FW_BLUETOOTH_STRING_UTILS_H_
#define _FW_BLUETOOTH_STRING_UTILS_H_

#include <string>

namespace fw {

/**
 * Check for valid Bluetooth device address.
 *
 * @param[in] address: Bluetooth device address
 * @param[in] withColon: flag indicating whether address has colon separators (default=false)
 * @param[in] allowEmpty: flag indicating whether empty address is allowed (default=false)
 *
 * @return = true: valid Bluetooth device address,
 * @return = false: invalid Bluetooth device address
 */
bool isValidBdAddress(const ::std::string& address, const bool withColon = false, const bool allowEmpty = false);

/**
 * Check for valid 128 bit UUID.
 *
 * @param[in] uuid: UUID
 * @param[in] withHyphen: flag indicating whether UUID has hyphen separators (default=false)
 * @param[in] allowEmpty: flag indicating whether empty UUID is allowed (default=false)
 *
 * @return = true: valid 128 bit UUID,
 * @return = false: invalid 128 bit UUID
 */
bool isValid128BitUuid(const ::std::string& uuid, const bool withHyphen = false, const bool allowEmpty = false);

/**
 * Check for Apple device based on given device identification data.
 *
 * @param[in] vendorId: vendor id
 * @param[in] vendorIdSource: vendor id source
 *
 * @return = true: Apple device,
 * @return = false: no Apple device
 */
bool isAppleDevice(const unsigned short int vendorId, const unsigned short int vendorIdSource);

/**
 * Get default Specification ID of Device ID Service Record.
 *
 * @return = default Specification ID of Device ID Service Record
 */
unsigned short int getDeviceIdSpecificationId(void);

/**
 * Get vendor id source for Bluetooth SIG.
 *
 * @return = vendor id source for Bluetooth SIG
 */
unsigned short int getVendorIdSourceBluetoothSIG(void);

/**
 * Get vendor id source for USB Implementer's Forum.
 *
 * @return = vendor id source for USB Implementer's Forum
 */
unsigned short int getVendorIdSourceUSBImplementersForum(void);

/**
 * Get vendor id source for Reserved.
 *
 * @return = vendor id source for Reserved
 */
unsigned short int getVendorIdSourceReserved(void);

/**
 * Get vendor id for Apple (Bluetooth SIG).
 *
 * @return = vendor id for Apple (Bluetooth SIG)
 */
unsigned short int getAppleVendorIdBluetoothSIG(void);

/**
 * Get vendor id for Apple (USB Implementer's Forum).
 *
 * @return = vendor id for Apple (USB Implementer's Forum)
 */
unsigned short int getAppleVendorIdUSBImplementersForum(void);

/**
 * Get iAP2 over Bluetooth UUID (outgoing).
 *
 * @return = iAP2 over Bluetooth UUID (outgoing)
 */
const ::std::string& getOutgoingIAP2Uuid(void);

/**
 * Get iAP2 over Bluetooth UUID (incoming).
 *
 * @return = iAP2 over Bluetooth UUID (incoming)
 */
const ::std::string& getIncomingIAP2Uuid(void);

/**
 * Get CarPlay wireless UUID (outgoing).
 *
 * @return = CarPlay wireless UUID (outgoing)
 */
const ::std::string& getOutgoingCPWUuid(void);

/**
 * Get CarPlay wireless UUID (incoming).
 *
 * @return = CarPlay wireless UUID (incoming)
 */
const ::std::string& getIncomingCPWUuid(void);

/**
 * Get general SPP UUID (outgoing).
 *
 * @return = general SPP UUID (outgoing)
 */
const ::std::string& getOutgoingSPPUuid(void);

/**
 * Get general SPP UUID (incoming).
 *
 * @return = general SPP UUID (incoming)
 */
const ::std::string& getIncomingSPPUuid(void);

/**
 * Get AndroidAuto wireless UUID (incoming).
 *
 * @return = AndroidAuto wireless UUID (incoming)
 */
const ::std::string& getIncomingAAWUuid(void);

/**
 * Get Base UUID without hyphen.
 *
 * @return = Base UUID without hyphen
 */
const ::std::string& getBaseUuidWithoutHyphen(void);

/**
 * Get Base UUID with hyphen.
 *
 * @return = Base UUID with hyphen
 */
const ::std::string& getBaseUuidWithHyphen(void);

} //fw

#endif //_FW_BLUETOOTH_STRING_UTILS_H_
