/**
 * @file FwArray.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief Array type.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Array type. This implementation shall avoid memory reallocation during every push_back() call of an vector<>.
 */

#ifndef _FW_ARRAY_H_
#define _FW_ARRAY_H_

#include <vector>
#include <stdbool.h> // <cstdbool> is marked with C++11

namespace fw {

/**
 * following is implemented:
 *    Array()
 *    ~Array()
 *    Array(const Array& rhs)
 *    Array(const ::std::vector<T>& rhs)
 *    Array& operator=(const Array& rhs)
 *    Array& operator=(const ::std::vector<T>& rhs)
 *    T& operator[](::std::size_t index)
 *    const T& operator[](::std::size_t index) const
 *    T& at(::std::size_t index)
 *    const T& at(::std::size_t index) const
 *    void push_back(const T& value)
 *    void pop_back(void)
 *    T& back(void)
 *    const T& back(void) const
 *    T& front(void)
 *    const T& front(void) const
 *    void clear(void)
 *    ::std::size_t size(void) const
 *    ::std::size_t capacity(void) const
 *    typename ::std::vector<T>::iterator begin(void)
 *    const typename ::std::vector<T>::iterator begin(void) const
 *    typename ::std::vector<T>::iterator end(void)
 *    const typename ::std::vector<T>::iterator end(void) const
 *    ::std::vector<T>& getData(void)
 *    const ::std::vector<T>& getData(void) const
 *    bool operator==(const Array& rhs) const
 *    bool operator!=(const Array& rhs) const
 *    bool operator==(const ::std::vector<T>& rhs) const
 *    bool operator!=(const ::std::vector<T>& rhs) const
 *    ::std::size_t getMinSize(void) const
 *
 * add more methods if needed
 */
template <typename T, ::std::size_t minSize>
class Array
{
public:
   /**
    * Default constructor.
    */
   Array()
   {
      // use given minimum size
      _minSize = minSize;
      // reserve at least 1 element
      if(0 == _minSize)
      {
         _minSize = 1;
      }
      // reserve memory
      _data.reserve(_minSize);
   }

   /**
    * Destructor.
    */
   ~Array()
   {
      // clear
      _data.clear();
   }

   /**
    * Copy constructor.
    *
    * @param[in] rhs: reference of input data
    */
   Array(const Array& rhs)
   {
      // use given minimum size
      _minSize = minSize;
      // reserve at least 1 element
      if(0 == _minSize)
      {
         _minSize = 1;
      }
      // reserve memory
      _data.reserve(_minSize);
      // copy data
      _data = rhs._data;
   }

   /**
    * Copy constructor.
    *
    * @param[in] rhs: reference of input data
    */
   Array(const ::std::vector<T>& rhs)
   {
      // use given minimum size
      _minSize = minSize;
      // reserve at least 1 element
      if(0 == _minSize)
      {
         _minSize = 1;
      }
      // reserve memory
      _data.reserve(_minSize);
      // copy data
      _data = rhs;
   }

   /**
    * Assignment operator.
    *
    * @param[in] rhs: reference of input data
    */
   Array& operator=(const Array& rhs)
   {
      if(this == &rhs)
      {
         return *this;
      }

      // copy data
      _data = rhs._data;

      return *this;
   }

   /**
    * Assignment operator.
    *
    * @param[in] rhs: reference of input data
    */
   Array& operator=(const ::std::vector<T>& rhs)
   {
      // it works (tested with debugger)
      if(&_data == &rhs)
      {
         return *this;
      }

      // copy data
      _data = rhs;

      return *this;
   }

   /**
    * [] operator.
    *
    * @param[in] index: index
    *
    * @return = reference to the element at position index
    */
   T& operator[](::std::size_t index)
   {
      if(_data.size() > index)
      {
         // in range
         return _data[index];
      }
      else
      {
         // out of range
         // we have reserved at least 1 element => return element at index 0
         return _data[0];
      }
   }

   /**
    * [] operator.
    *
    * @param[in] index: index
    *
    * @return = reference to the element at position index
    */
   const T& operator[](::std::size_t index) const
   {
      if(_data.size() > index)
      {
         // in range
         return _data[index];
      }
      else
      {
         // out of range
         // we have reserved at least 1 element => return element at index 0
         return _data[0];
      }
   }

   /**
    * at function.
    *
    * @param[in] index: index
    *
    * @return = reference to the element at position index
    */
   T& at(::std::size_t index)
   {
      if(_data.size() > index)
      {
         // in range
         return _data.at(index);
      }
      else
      {
         // out of range
         // we have reserved at least 1 element => return element at index 0
         return _data.at(0);
      }
   }

   /**
    * at function.
    *
    * @param[in] index: index
    *
    * @return = reference to the element at position index
    */
   const T& at(::std::size_t index) const
   {
      if(_data.size() > index)
      {
         // in range
         return _data.at(index);
      }
      else
      {
         // out of range
         // we have reserved at least 1 element => return element at index 0
         return _data.at(0);
      }
   }

   /**
    * push_back function (add element at the end).
    *
    * @param[in] value: value
    */
   void push_back(const T& value)
   {
      _data.push_back(value);
   }

   /**
    * pop_back function (delete last element).
    */
   void pop_back(void)
   {
      if(_data.size() > 0)
      {
         _data.pop_back();
      }
   }

   /**
    * back function (access last element).
    *
    * @return = reference to the last element
    */
   T& back(void)
   {
      return _data.back();
   }

   /**
    * back function (access last element).
    *
    * @return = reference to the last element
    */
   const T& back(void) const
   {
      return _data.back();
   }

   /**
    * front function (access first element).
    *
    * @return = reference to the first element
    */
   T& front(void)
   {
      return _data.front();
   }

   /**
    * front function (access first element).
    *
    * @return = reference to the first element
    */
   const T& front(void) const
   {
      return _data.front();
   }

   /**
    * clear function (removes all elements).
    */
   void clear(void)
   {
      _data.clear();
      // check capacity
      if(_data.capacity() < _minSize)
      {
         // reserve memory
         _data.reserve(_minSize);
      }
   }

   /**
    * size function (returns the number of elements).
    *
    * @return = number of elements
    */
   ::std::size_t size(void) const
   {
      return _data.size();
   }

   /**
    * capacity function (returns the size of the storage space currently allocated).
    *
    * @return = size of the storage space currently allocated
    */
   ::std::size_t capacity(void) const
   {
      return _data.capacity();
   }

   /**
    * begin function (returns an iterator pointing to the first element).
    *
    * @return = reference to the first element
    */
   typename ::std::vector<T>::iterator begin(void)
   {
      return _data.begin();
   }

   /**
    * begin function (returns an iterator pointing to the first element).
    *
    * @return = reference to the first element
    */
   const typename ::std::vector<T>::iterator begin(void) const
   {
      return _data.begin();
   }

   /**
    * end function (returns an iterator referring to the past-the-end element).
    *
    * @return = reference to the first element
    */
   typename ::std::vector<T>::iterator end(void)
   {
      return _data.end();
   }

   /**
    * end function (returns an iterator referring to the past-the-end element).
    *
    * @return = reference to the first element
    */
   const typename ::std::vector<T>::iterator end(void) const
   {
      return _data.end();
   }

   /**
    * Get vector.
    *
    * @return = reference to vector
    */
   ::std::vector<T>& getData(void)
   {
      return _data;
   }

   /**
    * Get vector.
    *
    * @return = reference to vector
    */
   const ::std::vector<T>& getData(void) const
   {
      return _data;
   }

   /**
    * == operator.
    *
    * @param[in] rhs: reference of input data
    *
    * @return = true: identical data,
    * @return = false: different data
    */
   bool operator==(const Array& rhs) const
   {
      // compare of _minSize is not needed
      return (_data == rhs._data);
   }

   /**
    * != operator.
    *
    * @param[in] rhs: reference of input data
    *
    * @return = false: identical data,
    * @return = true: different data
    */
   bool operator!=(const Array& rhs) const
   {
      return !(operator==(rhs));
   }

   /**
    * == operator.
    *
    * @param[in] rhs: reference of input data
    *
    * @return = true: identical data,
    * @return = false: different data
    */
   bool operator==(const ::std::vector<T>& rhs) const
   {
      // compare of _minSize is not needed
      return (_data == rhs._data);
   }

   /**
    * != operator.
    *
    * @param[in] rhs: reference of input data
    *
    * @return = false: identical data,
    * @return = true: different data
    */
   bool operator!=(const ::std::vector<T>& rhs) const
   {
      return !(operator==(rhs));
   }

   /**
    * get minimum size.
    *
    * @return = minimum size
    */
   ::std::size_t getMinSize(void) const
   {
      return _minSize;
   }

private:
   ::std::vector<T> _data;
   ::std::size_t _minSize;
};

} //fw

#endif //_FW_ARRAY_H_
