/**
 * @defgroup Dispatcher Dispatcher
 * @author Dinesh D
 *
 * Public interface to send a message to any registered Statemachine
 * @{
 */


#ifndef DISPATCHER_H_
#define DISPATCHER_H_

#include "SMF.h"
#include "Lock.h"
#ifdef VARIANT_S_FTR_ENABLE_FW_REMOTE_SM_MANAGER
#include "RemoteStateMachineManager.h"
#endif

/** 
 * Class to dispatch a message to any registered state machines from any component 
 */ 
class Dispatcher 	 
{

public:

    Dispatcher();
    ~Dispatcher();

    /**
     * Creates singleton object for Dispatcher
     *
     * @return pointer to static instance of Dispatcher object
     */
    static Dispatcher &GetInstance(void);

    /**
     * Register a statemachine object with the dispatcher
     *
     * @param[in] stateMachine pointer to state machine object
     *
     * @return < 0: error, = 0: OK
     */
    int Register( const SMF* stateMachine);

    /**
     * DeRegister a statemachine object from the dispatcher
     *
     * @param[in] stateMachine pointer to state machine object which was registered
     *
     * @return < 0: error, = 0: OK
     */
    int DeRegister( const SMF* stateMachine);

    /**
     * DeRegister all the  statemachine instances from the dispatcher
     *
     * @return < 0: error, = 0: OK
     */
    int DeRegisterAll( void );

    /**
     * Send a message to a statemachine as described in the message
     *
     * @param[in] message combination of destination Statemachine name and the actual message which
     *                       needs to be sent separated by "::"  ( e.g. SPM::InitReady )
     *
     * @return < 0: error, = 0: OK
     */
    int SendMessage(const char *message );
    int SendMessage(const char *message, const char *parameter );

    /**
     * Send a message to a statemachine as described in the message and expect the answer in return
     * when action is completed
     *
     * @param[in] message combination of destination Statemachine name and the actual message which
     *                       needs to be sent separated by "::"  ( e.g. SPM::InitReady ) and followed by parameters
     *                       seperated by ","
     * @param[in] answer  combination of client statemachine name which expects the answer and
     *                       the expected answer message in return. Both seperated by "::"
     *
     * @return < 0: error, = 0: OK
     */
    int SendMessageAnswer(const char *message, const char *answer);
    int SendMessageAnswer(const char *message, const char *parameter, const char *answer);
    int SendMessageAnswerLocal(const char *message, const char *answer);

    void ForceNoDestruction();

#ifdef VARIANT_S_FTR_ENABLE_FW_REMOTE_SM_MANAGER
    /* this must be public because the MediaEngine(Test) wants to start the Dbus main loop by using this object */
    RemoteStateMachineManager mRemoteStateMachineManager;
#endif

private:

    vector <SMF *> mActiveStateMachines;    /**< list of registered State machine objects*/
    Lock mActiveStateMachinesLock;
    int mDestructionFlag;

    SMF* GetSMHandle(const char* name);
    int SendMessageAnswerInternal(const char *message ,const char *answer, const int localOnly);
};

#endif /*DISPATCHER_H_*/
/** @} */
