/**
 * @file WblDbusParser.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief WBL DBUS Property Parser.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details WBL DBUS Property Parser.
 */

#include "WblDbusParser.h"
#include "WblDbusInterfaces.h"
#include "DbusParser.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblDbusParser.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace wbl {

static WblDbusInterfaces dbusInterfaces;
static DbusParser dbusParser;

WblDbusParser::WblDbusParser() :
_tracesEnabled(false)
{
   dbusParser.setTraces(_tracesEnabled);
}

WblDbusParser::~WblDbusParser()
{
   setTraces(false);
}

void WblDbusParser::parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseProperties(outProperties, matchingInterfaces, inProperties, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void WblDbusParser::parseIntrospection(const ::std::string& data,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseIntrospection(data, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void WblDbusParser::parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseInterfacesAdded(addedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void WblDbusParser::parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseInterfacesRemoved(removedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

bool WblDbusParser::findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface) const
{
   return dbusParser.findAndParseProperty(outPropertyList, inProperties, propertyName, interface, &dbusInterfaces);
}

void WblDbusParser::parseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface) const
{
   dbusParser.parsePropertyValue(outPropertyList, propertyValue, propertyName, interface, &dbusInterfaces);
}

const ::std::string& WblDbusParser::getRootObjectPath(void) const
{
   return dbusInterfaces.getRootObjectPath();
}

const ::std::string& WblDbusParser::getInterfaceName(const Interface interface) const
{
   return dbusInterfaces.getInterface2Name(interface);
}

const ::std::string& WblDbusParser::getObjectPath(const Interface interface) const
{
   return dbusInterfaces.getInterface2ObjectPath(interface);
}

const ::std::string& WblDbusParser::getBusName(const Interface interface) const
{
   return dbusInterfaces.getInterface2BusName(interface);
}

DbusBusType WblDbusParser::getBusType(const Interface interface) const
{
   return dbusInterfaces.getInterface2BusType(interface);
}

const ::std::string& WblDbusParser::getPort(const Interface interface) const
{
   return dbusInterfaces.getInterface2Port(interface);
}

LastIntendedModeProperty WblDbusParser::getLastIntendedModeProperty2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getLastIntendedModeProperty2Enum(value);
}

const ::std::string& WblDbusParser::getLastIntendedModeProperty2String(const LastIntendedModeProperty value) const
{
   return dbusInterfaces.getLastIntendedModeProperty2String(value);
}

PowerState WblDbusParser::getPowerState2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getPowerState2Enum(value);
}

const ::std::string& WblDbusParser::getPowerState2String(const PowerState value) const
{
   return dbusInterfaces.getPowerState2String(value);
}

WiFiMode WblDbusParser::getWiFiMode2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getWiFiMode2Enum(value);
}

const ::std::string& WblDbusParser::getWiFiMode2String(const WiFiMode value) const
{
   return dbusInterfaces.getWiFiMode2String(value);
}

WiFiFrequency WblDbusParser::getWiFiFrequency2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getWiFiFrequency2Enum(value);
}

const ::std::string& WblDbusParser::getWiFiFrequency2String(const WiFiFrequency value) const
{
   return dbusInterfaces.getWiFiFrequency2String(value);
}

WiFiType WblDbusParser::getWiFiType2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getWiFiType2Enum(value);
}

const ::std::string& WblDbusParser::getWiFiType2String(const WiFiType value) const
{
   return dbusInterfaces.getWiFiType2String(value);
}

SupportedSetting WblDbusParser::getSupportedSetting2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getSupportedSetting2Enum(value);
}

const ::std::string& WblDbusParser::getSupportedSetting2String(const SupportedSetting value) const
{
   return dbusInterfaces.getSupportedSetting2String(value);
}

void WblDbusParser::setTraces(const bool enabled)
{
   _tracesEnabled = enabled;
   dbusParser.setTraces(_tracesEnabled);
}

} //wbl
} //ccdbusif
