/**
 * @file WblDbusInterfaces.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief WBL DBUS Interfaces.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details WBL DBUS Interfaces.
 */

#include "WblDbusInterfaces.h"
#include "FwAssert.h"

namespace ccdbusif {
namespace wbl {

WblDbusInterfaces::WblDbusInterfaces() :
_dummy(),

_objPathRoot("/"),

_interfaceLastIntendedMode("org.bosch.wbl.LastIntendedMode"),
_interfaceConflictManagement("org.bosch.wbl.conflictmanagement"),
_interfaceWifiSetup("org.bosch.wbl.wifisetup"),

_objPathLastIntendedMode("/org/bosch/wbl/LastIntendedMode"),
_objPathConflictManagement("/org/bosch/wbl/conflictmanagement"),
_objPathWifiSetup("/org/bosch/wbl/wifisetup"),

_busNameLastIntendedMode("org.bosch.Wbl"),
_busNameConflictManagement("org.bosch.Wbl"),
_busNameWifiSetup("org.bosch.Wbl"),

_busTypeLastIntendedMode(BUS_TYPE_SYSTEM),
_busTypeConflictManagement(BUS_TYPE_SYSTEM),
_busTypeWifiSetup(BUS_TYPE_SYSTEM),

_portLastIntendedMode("wblLastIntendedModePort"),
_portConflictManagement("wblConflictManagementPort"),
_portWifiSetup("wblWifiSetupPort"),

_propertiesLastIntendedMode(),
_propertiesConflictManagement(),
_propertiesWifiSetup()
{
}

WblDbusInterfaces::~WblDbusInterfaces()
{
}

void WblDbusInterfaces::getInterfaces(::std::vector<const ::std::map< std::string, int >*>& interfaceList, const ::std::vector< int >& matchingInterfaces) const
{
   interfaceList.reserve(matchingInterfaces.size());

   for(size_t i = 0; i < matchingInterfaces.size(); i++)
   {
      switch((Interface)matchingInterfaces[i])
      {
         case IF_LAST_INTENDED_MODE:
            interfaceList.push_back(_propertiesLastIntendedMode.getString2EnumListPtr());
            break;
         case IF_CONFLICT_MANAGEMENT:
            interfaceList.push_back(_propertiesConflictManagement.getString2EnumListPtr());
            break;
         case IF_WIFI_SETUP:
            interfaceList.push_back(_propertiesWifiSetup.getString2EnumListPtr());
            break;
         case IF_UNKNOWN:
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
}

const ::std::string& WblDbusInterfaces::getRootObjectPath(void) const
{
   return _objPathRoot;
}

const ::std::string& WblDbusInterfaces::getInterface2Name(const Interface value) const
{
   switch(value)
   {
      case IF_LAST_INTENDED_MODE:
         return _interfaceLastIntendedMode;
      case IF_CONFLICT_MANAGEMENT:
         return _interfaceConflictManagement;
      case IF_WIFI_SETUP:
         return _interfaceWifiSetup;
      case IF_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return _dummy;
   }
}

const ::std::string& WblDbusInterfaces::getInterface2ObjectPath(const Interface value) const
{
   switch(value)
   {
      case IF_LAST_INTENDED_MODE:
         return _objPathLastIntendedMode;
      case IF_CONFLICT_MANAGEMENT:
         return _objPathConflictManagement;
      case IF_WIFI_SETUP:
         return _objPathWifiSetup;
      case IF_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return _dummy;
   }
}

const ::std::string& WblDbusInterfaces::getInterface2BusName(const Interface value) const
{
   switch(value)
   {
      case IF_LAST_INTENDED_MODE:
         return _busNameLastIntendedMode;
      case IF_CONFLICT_MANAGEMENT:
         return _busNameConflictManagement;
      case IF_WIFI_SETUP:
         return _busNameWifiSetup;
      case IF_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return _dummy;
   }
}

DbusBusType WblDbusInterfaces::getInterface2BusType(const Interface value) const
{
   switch(value)
   {
      case IF_LAST_INTENDED_MODE:
         return _busTypeLastIntendedMode;
      case IF_CONFLICT_MANAGEMENT:
         return _busTypeConflictManagement;
      case IF_WIFI_SETUP:
         return _busTypeWifiSetup;
      case IF_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return BUS_TYPE_SYSTEM;
   }
}

const ::std::string& WblDbusInterfaces::getInterface2Port(const Interface value) const
{
   switch(value)
   {
      case IF_LAST_INTENDED_MODE:
         return _portLastIntendedMode;
      case IF_CONFLICT_MANAGEMENT:
         return _portConflictManagement;
      case IF_WIFI_SETUP:
         return _portWifiSetup;
      case IF_UNKNOWN:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         return _dummy;
   }
}

} //wbl
} //ccdbusif
