/**
 * @file IWblDbusConflictManagementSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS Send Request Interface for ConflictManagement.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for Wbl DBUS send request for ConflictManagement.
 */

#ifndef _I_WBL_DBUS_CONFLICT_MANAGEMENT_SEND_REQUEST_IF_H_
#define _I_WBL_DBUS_CONFLICT_MANAGEMENT_SEND_REQUEST_IF_H_

#include "CcDbusIfTypes.h"

namespace ccdbusif {

// class forward declarations
class IDestroyAllProxies;
class IWblDbusConflictManagementCallbackIf;

/**
 * Send request interface class for ConflictManagement.
 */
class IWblDbusConflictManagementSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblDbusConflictManagementSendRequestIf() {}

   /**
    * Get destroy interface.
    *
    * @return = destroy interface
    */
   virtual IDestroyAllProxies* getDestroyer(void) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void setCallbackIf(IWblDbusConflictManagementCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Create proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] createDirectly: flag to trigger direct proxy creation or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxyIf(const unsigned int callbackId, const bool createDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxyIf(const unsigned int callbackId, const bool destroyDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Send PrepareSetup Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] wiFiMode: WiFi mode (NOTE: this is only one of the possible parameters; if interface is needed then extend)
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPrepareSetupRequest(const unsigned int callbackId, const ::std::string& wiFiMode) = 0;

   /**
    * Send DeActivateSetup Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] wiFiMode: WiFi mode
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeActivateSetupRequest(const unsigned int callbackId, const ::std::string& wiFiMode) = 0;

   /**
    * Send WiFiStationModeUsage Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] application: application
    * @param[in] usage: usage e.g. download
    * @param[in] holding: started using / finished flag
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWiFiStationModeUsageRequest(const unsigned int callbackId, const ::std::string& application, const ::std::string& usage, bool holding) = 0;

   /**
    * Send ResolveConflict Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] conflictid: conflictid
    * @param[in] action: action
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendResolveConflictRequest(const unsigned int callbackId, int32 conflictid, int32 action) = 0;

   /**
    * Send ActiveWiFiSetups Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendActiveWiFiSetupsGet(const unsigned int callbackId) = 0;

   /**
    * Send WiFiPriorityList Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWiFiPriorityListGet(const unsigned int callbackId) = 0;

   /**
    * Send WiFiPriorityList Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] wiFiPriorityList: WiFi priority list
    */
   virtual void sendWiFiPriorityListSet(const unsigned int callbackId, const ::std::vector< ::std::string >& wiFiPriorityList) = 0;

   /**
    * Send ConfigurationTimer Get.
    *
    * @param[in] callbackId: callback id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConfigurationTimerGet(const unsigned int callbackId) = 0;

   /**
    * Send ConfigurationTimer Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] configurationTimer: configuration timer
    */
   virtual void sendConfigurationTimerSet(const unsigned int callbackId, int32 configurationTimer) = 0;
};

} //ccdbusif

#endif //_I_WBL_DBUS_CONFLICT_MANAGEMENT_SEND_REQUEST_IF_H_
