/**
 * @file IntrospectableDbusProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Introspectable DBUS Proxy.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Introspectable DBUS Proxy.
 */

#include "IntrospectableDbusProxyIf.h"
#include "IIntrospectableDbusCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "IntrospectableCreateRuntimeProxyWorkItem.h"
#include "IntrospectableDestroyRuntimeProxyWorkItem.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::introspect::org::freedesktop::DBus::Introspectable;

using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/IntrospectableDbusProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

IntrospectableDbusProxyIf::IntrospectableDbusProxyIf()
{
   // _proxy
}

IntrospectableDbusProxyIf::IntrospectableDbusProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _proxy
}

IntrospectableDbusProxyIf::~IntrospectableDbusProxyIf()
{
}

void IntrospectableDbusProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Introspectable"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxy.resetAllProxiesAndClear();
   _proxy.removeAllCallbackIfs();
}

void IntrospectableDbusProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Introspectable"));

   // HINT: feature is currently not used; implement if necessary
}

void IntrospectableDbusProxyIf::createRuntimeProxy(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user, IIntrospectableDbusCallbackIf* callbackIf)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   if(false == _proxy.isInstanceAvailable(metaData))
   {
      ::boost::shared_ptr< IntrospectableProxy > introspectableProxy;

      // next check is only necessary because multiple clients can use same proxy
      if(false == _proxy.isProxyAvailable(introspectableProxy, metaData))
      {
         introspectableProxy = IntrospectableProxy::createProxy("allIntrospectablePort", busName, objPath, convertLocalBusType(busType), *this);
      }
      _proxy.addProxyInstance(metaData, introspectableProxy);
      _proxy.addCallbackIf(metaData, callbackIf);
   }
}

void IntrospectableDbusProxyIf::destroyRuntimeProxy(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   // nothing to deregister
   _proxy.removeProxyInstance(metaData);
   _proxy.removeCallbackIf(metaData);
}

void IntrospectableDbusProxyIf::setCallbackIf(IIntrospectableDbusCallbackIf* callbackIf, bool enableProxy, const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(enableProxy);

   // callback interface is now available => create proxy now
   if((NULL != callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new IntrospectableCreateRuntimeProxyWorkItem(busName, objPath, busType, user, callbackIf, this));
   }
}

// ServiceAvailableIF implementation --- start
void IntrospectableDbusProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   for(unsigned int i = 0; i < _proxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< IntrospectableProxy >& introspectableProxy = _proxy[i];

      if((NULL != introspectableProxy.get()) && (proxy == introspectableProxy))
      {
         const ::std::string& busName = introspectableProxy->getDBusBusName();
         const ::std::string& objectPath = introspectableProxy->getDBusObjectPath();
         DbusBusType busType = convertDBusBusType(introspectableProxy->getBusType());

         IIntrospectableDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objectPath, busType, 0);

         if(NULL != callback)
         {
            // nothing to register

            callback->onIntrospectableProxyAvailable(busName, objectPath, busType, 0, stateChange.getPreviousState(), stateChange.getCurrentState());
         }
         break;
      }
   }
}

void IntrospectableDbusProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   for(unsigned int i = 0; i < _proxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< IntrospectableProxy >& introspectableProxy = _proxy[i];

      if((NULL != introspectableProxy.get()) && (proxy == introspectableProxy))
      {
         const ::std::string& busName = introspectableProxy->getDBusBusName();
         const ::std::string& objectPath = introspectableProxy->getDBusObjectPath();
         DbusBusType busType = convertDBusBusType(introspectableProxy->getBusType());

         IIntrospectableDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objectPath, busType, 0);

         // nothing to deregister

         if(NULL != callback)
         {
            callback->onIntrospectableProxyUnavailable(busName, objectPath, busType, 0, stateChange.getPreviousState(), stateChange.getCurrentState());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IIntrospectableDbusSendRequestIf implementation --- start
act_t IntrospectableDbusProxyIf::sendIntrospectRequest(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   ::boost::shared_ptr< IntrospectableProxy > introspectableProxy;
   if(true == _proxy.isProxyAvailable(introspectableProxy, metaData))
   {
      IIntrospectableDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objPath, busType, user);
      if(NULL != callback)
      {
         return introspectableProxy->sendIntrospectRequest(*callback); // TODO: this will not work if callback/entry is removed before answer is received; valid for all dynamic interfaces
      }
   }

   return DEFAULT_ACT;
}
// IIntrospectableDbusSendRequestIf implementation --- end

} //ccdbusif
