/**
 * @file EvolutionGeniviUtility.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi Utility.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi Utility.
 */

#include "EvolutionGeniviUtility.h"

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

#include <cstdio>
#include <cstring>

namespace ccdbusif {
namespace evolution_genivi_utility {

static const ::std::string pbapInfoFormat = "version:0x%x,support_repositories:0x%x,support_feature:0x%x";
static const ::std::string mapInfoFormatPart1 = "(remote_instance:0x%x,version:0x%x,support_type:0x%x,name:";
static const ::std::string mapInfoFormatPart3 = ",support_feature:0x%x,reserved:0x%x)";

bool extractPbapData(unsigned short int& version, unsigned char& repositories, unsigned int& features, const ::std::string& data)
{
   // string "version:0x102,support_repositories:0xb,support_feature:0x3ff"

   if(true == data.empty())
   {
      // invalid data
      return false;
   }

   unsigned int data0 = 0;
   unsigned int data1 = 0;
   unsigned int data2 = 0;

   int nmbItems = sscanf(data.c_str(), pbapInfoFormat.c_str(), &data0, &data1, &data2);

   if(3 == nmbItems)
   {
      if((0 != data0) || (0 != data1) || (0 != data2))
      {
         version = (unsigned short int)data0;
         repositories = (unsigned char)data1;
         features = data2;
         return true;
      }
      else
      {
         // all values are 0 => invalid data
         return false;
      }
   }
   else
   {
      // wrong number of extracted items
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }
}

bool extractMapData(unsigned short int& version, unsigned char& instance, ::std::string& name, unsigned char& types, unsigned int& features, const ::std::string& data)
{
   // string "(remote_instance:0x0,version:0x102,support_type:0xe,name:RPI MAS Instance No 0,support_feature:0x7f,reserved:0x0)"

   if(true == data.empty())
   {
      // invalid data
      return false;
   }

   unsigned int data0 = 0;
   unsigned int data1 = 0;
   unsigned int data2 = 0;
   char data3[1 + data.size()];
   unsigned int data4 = 0;
   unsigned int data5 = 0;
   const char* left;
   const char* right;
   char* dest;

   // parse instance, version, types
   left = data.c_str();
   right = left;
   dest = data3;
   while(('\0' != *right) && (',' != *right))
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   if('\0' != *right)
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   while(('\0' != *right) && (',' != *right))
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   if('\0' != *right)
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   while(('\0' != *right) && (',' != *right))
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   if('\0' != *right)
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   while(('\0' != *right) && (':' != *right))
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   if('\0' != *right)
   {
      *dest = *right;
      ++dest;
      ++right;
   }
   *dest = '\0';

   int c = sscanf(data3, mapInfoFormatPart1.c_str(), &data0, &data1, &data2);

   if(3 != c)
   {
      // wrong number of extracted items
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   // parse features, reserved
   left = right; // start of name
   right = data.c_str() + data.size(); // NULL termination
   while((left != right) && (',' != *right))
   {
      --right;
   }
   if(left != right)
   {
      --right;
   }
   while((left != right) && (',' != *right))
   {
      --right;
   }

   if(left > right)
   {
      // something went wrong
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   c = sscanf(right, mapInfoFormatPart3.c_str(), &data4, &data5);

   if(2 != c)
   {
      // wrong number of extracted items
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   // parse name
   const size_t len = (size_t)(right - left);

   if(len > data.size())
   {
      // something went wrong
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   strncpy(data3, left, len);
   data3[len] = '\0';

   if((0 != data0) || (0 != data1) || (0 != data2) || ('\0' != data3[0]) || (0 != data4) || (0 != data5))
   {
      instance = (unsigned char)data0;
      version = (unsigned short int)data1;
      types = (unsigned char)data2;
      name = data3;
      features = data4;
      // data5 not used at the moment

      return true;
   }

   return false;
}

//******************************************************************************
// HINT: all data types are stored in big endian format
//******************************************************************************
void writeInt8(::std::vector<unsigned char>& array, const int8_t data)
{
   array.push_back((unsigned char)data);
}

void writeUInt8(::std::vector<unsigned char>& array, const uint8_t data)
{
   array.push_back(data);
}

void writeInt16(::std::vector<unsigned char>& array, const int16_t data)
{
   uint16_t temp = (uint16_t)data;
   array.push_back((unsigned char)((temp >> 8) & 0x00ff));
   array.push_back((unsigned char)((temp     ) & 0x00ff));
}

void writeUInt16(::std::vector<unsigned char>& array, const uint16_t data)
{
   array.push_back((unsigned char)((data >> 8) & 0x00ff));
   array.push_back((unsigned char)((data     ) & 0x00ff));
}

void writeInt32(::std::vector<unsigned char>& array, const int32_t data)
{
   uint32_t temp = (uint32_t)data;
   array.push_back((unsigned char)((temp >> 24) & 0x000000ffL));
   array.push_back((unsigned char)((temp >> 16) & 0x000000ffL));
   array.push_back((unsigned char)((temp >>  8) & 0x000000ffL));
   array.push_back((unsigned char)((temp      ) & 0x000000ffL));
}

void writeUInt32(::std::vector<unsigned char>& array, const uint32_t data)
{
   array.push_back((unsigned char)((data >> 24) & 0x000000ffL));
   array.push_back((unsigned char)((data >> 16) & 0x000000ffL));
   array.push_back((unsigned char)((data >>  8) & 0x000000ffL));
   array.push_back((unsigned char)((data      ) & 0x000000ffL));
}

void writeInt64(::std::vector<unsigned char>& array, const int64_t data)
{
   uint64_t temp = (uint64_t)data;
   array.push_back((unsigned char)((temp >> 56) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >> 48) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >> 40) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >> 32) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >> 24) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >> 16) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp >>  8) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((temp      ) & 0x00000000000000ffLL));
}

void writeUInt64(::std::vector<unsigned char>& array, const uint64_t data)
{
   array.push_back((unsigned char)((data >> 56) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >> 48) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >> 40) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >> 32) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >> 24) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >> 16) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data >>  8) & 0x00000000000000ffLL));
   array.push_back((unsigned char)((data      ) & 0x00000000000000ffLL));
}

void writeString(::std::vector<unsigned char>& array, const ::std::string& data)
{
   for(size_t i = 0; i < data.size(); i++)
   {
      array.push_back((unsigned char)data[i]);
   }
}

void writeArray(::std::vector<unsigned char>& array, const ::std::vector<unsigned char>& data)
{
   array.insert(array.end(), data.begin(), data.end());
}

void readUInt16(uint16_t& data, const ::std::vector<unsigned char>& array)
{
   if(2 == array.size())
   {
      data = (uint16_t)((array[0] << 8) | (array[1]));
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      data = 0;
   }
}

} //evolution_genivi_utility
} //ccdbusif
