/**
 * @file EvolutionGeniviDbusUuids.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Properties for UUIDs.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Properties for UUIDs.
 */

#include "EvolutionGeniviDbusUuids.h"
#include "FwStringUtils.h"

#include <cstdio>

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusUuids.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace evolution {

EvolutionGeniviDbusUuids::EvolutionGeniviDbusUuids() : DbusProperties("UUIDs"),
   _uuid32("00000000"),

   _uuidBase("00000000-0000-1000-8000-00805f9b34fb"),

   _uuidSDP("00000001-0000-1000-8000-00805f9b34fb"),
   _uuidUDP("00000002-0000-1000-8000-00805f9b34fb"),
   _uuidRFCOMM("00000003-0000-1000-8000-00805f9b34fb"),
   _uuidTCP("00000004-0000-1000-8000-00805f9b34fb"),
   _uuidTCS_BIN("00000005-0000-1000-8000-00805f9b34fb"),
   _uuidTCS_AT("00000006-0000-1000-8000-00805f9b34fb"),
   _uuidATT("00000007-0000-1000-8000-00805f9b34fb"),
   _uuidOBEX("00000008-0000-1000-8000-00805f9b34fb"),
   _uuidIP("00000009-0000-1000-8000-00805f9b34fb"),
   _uuidFTP("0000000a-0000-1000-8000-00805f9b34fb"),
   _uuidHTTP("0000000c-0000-1000-8000-00805f9b34fb"),
   _uuidWSP("0000000e-0000-1000-8000-00805f9b34fb"),
   _uuidBNEP("0000000f-0000-1000-8000-00805f9b34fb"),
   _uuidUPNP("00000010-0000-1000-8000-00805f9b34fb"),
   _uuidHIDP("00000011-0000-1000-8000-00805f9b34fb"),
   _uuidHardcopyControlChannel("00000012-0000-1000-8000-00805f9b34fb"),
   _uuidHardcopyDataChannel("00000014-0000-1000-8000-00805f9b34fb"),
   _uuidHardcopyNotification("00000016-0000-1000-8000-00805f9b34fb"),
   _uuidAVCTP("00000017-0000-1000-8000-00805f9b34fb"),
   _uuidAVDTP("00000019-0000-1000-8000-00805f9b34fb"),
   _uuidCMTP("0000001b-0000-1000-8000-00805f9b34fb"),
   _uuidMCAPControlChannel("0000001e-0000-1000-8000-00805f9b34fb"),
   _uuidMCAPDataChannel("0000001f-0000-1000-8000-00805f9b34fb"),
   _uuidL2CAP("00000100-0000-1000-8000-00805f9b34fb"),

   _uuidServiceDiscoveryServerServiceClassID("00001000-0000-1000-8000-00805f9b34fb"),
   _uuidBrowseGroupDescriptorServiceClassID("00001001-0000-1000-8000-00805f9b34fb"),
   _uuidSerialPort("00001101-0000-1000-8000-00805f9b34fb"),
   _uuidLANAccessUsingPPP("00001102-0000-1000-8000-00805f9b34fb"),
   _uuidDialupNetworking("00001103-0000-1000-8000-00805f9b34fb"),
   _uuidIrMCSync("00001104-0000-1000-8000-00805f9b34fb"),
   _uuidOBEXObjectPush("00001105-0000-1000-8000-00805f9b34fb"),
   _uuidOBEXFileTransfer("00001106-0000-1000-8000-00805f9b34fb"),
   _uuidIrMCSyncCommand("00001107-0000-1000-8000-00805f9b34fb"),
   _uuidHeadset("00001108-0000-1000-8000-00805f9b34fb"),
   _uuidCordlessTelephony("00001109-0000-1000-8000-00805f9b34fb"),
   _uuidAudioSource("0000110a-0000-1000-8000-00805f9b34fb"),
   _uuidAudioSink("0000110b-0000-1000-8000-00805f9b34fb"),
   _uuidA_V_RemoteControlTarget("0000110c-0000-1000-8000-00805f9b34fb"),
   _uuidAdvancedAudioDistribution("0000110d-0000-1000-8000-00805f9b34fb"),
   _uuidA_V_RemoteControl("0000110e-0000-1000-8000-00805f9b34fb"),
   _uuidA_V_RemoteControlController("0000110f-0000-1000-8000-00805f9b34fb"),
   _uuidIntercom("00001110-0000-1000-8000-00805f9b34fb"),
   _uuidFax("00001111-0000-1000-8000-00805f9b34fb"),
   _uuidHeadset_Audio_Gateway("00001112-0000-1000-8000-00805f9b34fb"),
   _uuidWAP("00001113-0000-1000-8000-00805f9b34fb"),
   _uuidWAP_CLIENT("00001114-0000-1000-8000-00805f9b34fb"),
   _uuidPANU("00001115-0000-1000-8000-00805f9b34fb"),
   _uuidNAP("00001116-0000-1000-8000-00805f9b34fb"),
   _uuidGN("00001117-0000-1000-8000-00805f9b34fb"),
   _uuidDirectPrinting("00001118-0000-1000-8000-00805f9b34fb"),
   _uuidReferencePrinting("00001119-0000-1000-8000-00805f9b34fb"),
   _uuidBasic_Imaging_Profile("0000111a-0000-1000-8000-00805f9b34fb"),
   _uuidImagingResponder("0000111b-0000-1000-8000-00805f9b34fb"),
   _uuidImagingAutomaticArchive("0000111c-0000-1000-8000-00805f9b34fb"),
   _uuidImagingReferencedObjects("0000111d-0000-1000-8000-00805f9b34fb"),
   _uuidHandsfree("0000111e-0000-1000-8000-00805f9b34fb"),
   _uuidHandsfreeAudioGateway("0000111f-0000-1000-8000-00805f9b34fb"),
   _uuidDirectPrintingReferenceObjectsService("00001120-0000-1000-8000-00805f9b34fb"),
   _uuidReflectedUI("00001121-0000-1000-8000-00805f9b34fb"),
   _uuidBasicPrinting("00001122-0000-1000-8000-00805f9b34fb"),
   _uuidPrintingStatus("00001123-0000-1000-8000-00805f9b34fb"),
   _uuidHumanInterfaceDeviceService("00001124-0000-1000-8000-00805f9b34fb"),
   _uuidHardcopyCableReplacement("00001125-0000-1000-8000-00805f9b34fb"),
   _uuidHCR_Print("00001126-0000-1000-8000-00805f9b34fb"),
   _uuidHCR_Scan("00001127-0000-1000-8000-00805f9b34fb"),
   _uuidCommon_ISDN_Access("00001128-0000-1000-8000-00805f9b34fb"),
   _uuidSIM_Access("0000112d-0000-1000-8000-00805f9b34fb"),
   _uuidPhonebook_Access_PCE("0000112e-0000-1000-8000-00805f9b34fb"),
   _uuidPhonebook_Access_PSE("0000112f-0000-1000-8000-00805f9b34fb"),
   _uuidPhonebook_Access("00001130-0000-1000-8000-00805f9b34fb"),
   _uuidHeadset_HS("00001131-0000-1000-8000-00805f9b34fb"),
   _uuidMessage_Access_Server("00001132-0000-1000-8000-00805f9b34fb"),
   _uuidMessage_Notification_Server("00001133-0000-1000-8000-00805f9b34fb"),
   _uuidMessage_Access_Profile("00001134-0000-1000-8000-00805f9b34fb"),
   _uuidGNSS("00001135-0000-1000-8000-00805f9b34fb"),
   _uuidGNSS_Server("00001136-0000-1000-8000-00805f9b34fb"),
   _uuid3D_Display("00001137-0000-1000-8000-00805f9b34fb"),
   _uuid3D_Glasses("00001138-0000-1000-8000-00805f9b34fb"),
   _uuid3D_Synchronization("00001139-0000-1000-8000-00805f9b34fb"),
   _uuidMPS_Profile("0000113a-0000-1000-8000-00805f9b34fb"),
   _uuidMPS_SC("0000113b-0000-1000-8000-00805f9b34fb"),
   _uuidCTN_Access_Service("0000113c-0000-1000-8000-00805f9b34fb"),
   _uuidCTN_Notification_Service("0000113d-0000-1000-8000-00805f9b34fb"),
   _uuidCTN_Profile("0000113e-0000-1000-8000-00805f9b34fb"),
   _uuidPnPInformation("00001200-0000-1000-8000-00805f9b34fb"),
   _uuidGenericNetworking("00001201-0000-1000-8000-00805f9b34fb"),
   _uuidGenericFileTransfer("00001202-0000-1000-8000-00805f9b34fb"),
   _uuidGenericAudio("00001203-0000-1000-8000-00805f9b34fb"),
   _uuidGenericTelephony("00001204-0000-1000-8000-00805f9b34fb"),
   _uuidUPNP_Service("00001205-0000-1000-8000-00805f9b34fb"),
   _uuidUPNP_IP_Service("00001206-0000-1000-8000-00805f9b34fb"),
   _uuidESDP_UPNP_IP_PAN("00001300-0000-1000-8000-00805f9b34fb"),
   _uuidESDP_UPNP_IP_LAP("00001301-0000-1000-8000-00805f9b34fb"),
   _uuidESDP_UPNP_L2CAP("00001302-0000-1000-8000-00805f9b34fb"),
   _uuidVideoSource("00001303-0000-1000-8000-00805f9b34fb"),
   _uuidVideoSink("00001304-0000-1000-8000-00805f9b34fb"),
   _uuidVideoDistribution("00001305-0000-1000-8000-00805f9b34fb"),
   _uuidHDP("00001400-0000-1000-8000-00805f9b34fb"),
   _uuidHDP_Source("00001401-0000-1000-8000-00805f9b34fb"),
   _uuidHDP_Sink("00001402-0000-1000-8000-00805f9b34fb"),

   _uuidSyncMLService("00000001-0000-1000-8000-0002ee000002"),

   _traceBASE("BASE"),
   _traceSDP("SDP"),
   _traceUDP("UDP"),
   _traceRFCOMM("RFCOMM"),
   _traceTCP("TCP"),
   _traceTCS_BIN("TCS_BIN"),
   _traceTCS_AT("TCS_AT"),
   _traceATT("ATT"),
   _traceOBEX("OBEX"),
   _traceIP("IP"),
   _traceFTP("FTP"),
   _traceHTTP("HTTP"),
   _traceWSP("WSP"),
   _traceBNEP("BNEP"),
   _traceUPNP("UPNP"),
   _traceHIDP("HIDP"),
   _traceHardcopyControlChannel("HardcopyControlChannel"),
   _traceHardcopyDataChannel("HardcopyDataChannel"),
   _traceHardcopyNotification("HardcopyNotification"),
   _traceAVCTP("AVCTP"),
   _traceAVDTP("AVDTP"),
   _traceCMTP("CMTP"),
   _traceMCAPControlChannel("MCAPControlChannel"),
   _traceMCAPDataChannel("MCAPDataChannel"),
   _traceL2CAP("L2CAP"),
   _traceServiceDiscoveryServerServiceClassID("ServiceDiscoveryServerServiceClassID"),
   _traceBrowseGroupDescriptorServiceClassID("BrowseGroupDescriptorServiceClassID"),
   _traceSerialPort("SerialPort"),
   _traceLANAccessUsingPPP("LANAccessUsingPPP"),
   _traceDialupNetworking("DialupNetworking"),
   _traceIrMCSync("IrMCSync"),
   _traceOBEXObjectPush("OBEXObjectPush"),
   _traceOBEXFileTransfer("OBEXFileTransfer"),
   _traceIrMCSyncCommand("IrMCSyncCommand"),
   _traceHeadset("Headset"),
   _traceCordlessTelephony("CordlessTelephony"),
   _traceAudioSource("AudioSource"),
   _traceAudioSink("AudioSink"),
   _traceA_V_RemoteControlTarget("A_V_RemoteControlTarget"),
   _traceAdvancedAudioDistribution("AdvancedAudioDistribution"),
   _traceA_V_RemoteControl("A_V_RemoteControl"),
   _traceA_V_RemoteControlController("A_V_RemoteControlController"),
   _traceIntercom("Intercom"),
   _traceFax("Fax"),
   _traceHeadset_Audio_Gateway("Headset_Audio_Gateway"),
   _traceWAP("WAP"),
   _traceWAP_CLIENT("WAP_CLIENT"),
   _tracePANU("PANU"),
   _traceNAP("NAP"),
   _traceGN("GN"),
   _traceDirectPrinting("DirectPrinting"),
   _traceReferencePrinting("ReferencePrinting"),
   _traceBasic_Imaging_Profile("Basic_Imaging_Profile"),
   _traceImagingResponder("ImagingResponder"),
   _traceImagingAutomaticArchive("ImagingAutomaticArchive"),
   _traceImagingReferencedObjects("ImagingReferencedObjects"),
   _traceHandsfree("Handsfree"),
   _traceHandsfreeAudioGateway("HandsfreeAudioGateway"),
   _traceDirectPrintingReferenceObjectsService("DirectPrintingReferenceObjectsService"),
   _traceReflectedUI("ReflectedUI"),
   _traceBasicPrinting("BasicPrinting"),
   _tracePrintingStatus("PrintingStatus"),
   _traceHumanInterfaceDeviceService("HumanInterfaceDeviceService"),
   _traceHardcopyCableReplacement("HardcopyCableReplacement"),
   _traceHCR_Print("HCR_Print"),
   _traceHCR_Scan("HCR_Scan"),
   _traceCommon_ISDN_Access("Common_ISDN_Access"),
   _traceSIM_Access("SIM_Access"),
   _tracePhonebook_Access_PCE("Phonebook_Access_PCE"),
   _tracePhonebook_Access_PSE("Phonebook_Access_PSE"),
   _tracePhonebook_Access("Phonebook_Access"),
   _traceHeadset_HS("Headset_HS"),
   _traceMessage_Access_Server("Message_Access_Server"),
   _traceMessage_Notification_Server("Message_Notification_Server"),
   _traceMessage_Access_Profile("Message_Access_Profile"),
   _traceGNSS("GNSS"),
   _traceGNSS_Server("GNSS_Server"),
   _trace3D_Display("3D_Display"),
   _trace3D_Glasses("3D_Glasses"),
   _trace3D_Synchronization("3D_Synchronization"),
   _traceMPS_Profile("MPS_Profile"),
   _traceMPS_SC("MPS_SC"),
   _traceCTN_Access_Service("CTN_Access_Service"),
   _traceCTN_Notification_Service("CTN_Notification_Service"),
   _traceCTN_Profile("CTN_Profile"),
   _tracePnPInformation("PnPInformation"),
   _traceGenericNetworking("GenericNetworking"),
   _traceGenericFileTransfer("GenericFileTransfer"),
   _traceGenericAudio("GenericAudio"),
   _traceGenericTelephony("GenericTelephony"),
   _traceUPNP_Service("UPNP_Service"),
   _traceUPNP_IP_Service("UPNP_IP_Service"),
   _traceESDP_UPNP_IP_PAN("ESDP_UPNP_IP_PAN"),
   _traceESDP_UPNP_IP_LAP("ESDP_UPNP_IP_LAP"),
   _traceESDP_UPNP_L2CAP("ESDP_UPNP_L2CAP"),
   _traceVideoSource("VideoSource"),
   _traceVideoSink("VideoSink"),
   _traceVideoDistribution("VideoDistribution"),
   _traceHDP("HDP"),
   _traceHDP_Source("HDP_Source"),
   _traceHDP_Sink("HDP_Sink"),
   _traceSYNCML("SYNCML")
{
   _string2EnumList[_uuidBase] = (int)UUID_BASE;
   _string2EnumList[_uuidSDP] = (int)UUID_SDP;
   _string2EnumList[_uuidUDP] = (int)UUID_UDP;
   _string2EnumList[_uuidRFCOMM] = (int)UUID_RFCOMM;
   _string2EnumList[_uuidTCP] = (int)UUID_TCP;
   _string2EnumList[_uuidTCS_BIN] = (int)UUID_TCS_BIN;
   _string2EnumList[_uuidTCS_AT] = (int)UUID_TCS_AT;
   _string2EnumList[_uuidATT] = (int)UUID_ATT;
   _string2EnumList[_uuidOBEX] = (int)UUID_OBEX;
   _string2EnumList[_uuidIP] = (int)UUID_IP;
   _string2EnumList[_uuidFTP] = (int)UUID_FTP;
   _string2EnumList[_uuidHTTP] = (int)UUID_HTTP;
   _string2EnumList[_uuidWSP] = (int)UUID_WSP;
   _string2EnumList[_uuidBNEP] = (int)UUID_BNEP;
   _string2EnumList[_uuidUPNP] = (int)UUID_UPNP;
   _string2EnumList[_uuidHIDP] = (int)UUID_HIDP;
   _string2EnumList[_uuidHardcopyControlChannel] = (int)UUID_HARDCOPYCONTROLCHANNEL;
   _string2EnumList[_uuidHardcopyDataChannel] = (int)UUID_HARDCOPYDATACHANNEL;
   _string2EnumList[_uuidHardcopyNotification] = (int)UUID_HARDCOPYNOTIFICATION;
   _string2EnumList[_uuidAVCTP] = (int)UUID_AVCTP;
   _string2EnumList[_uuidAVDTP] = (int)UUID_AVDTP;
   _string2EnumList[_uuidCMTP] = (int)UUID_CMTP;
   _string2EnumList[_uuidMCAPControlChannel] = (int)UUID_MCAPCONTROLCHANNEL;
   _string2EnumList[_uuidMCAPDataChannel] = (int)UUID_MCAPDATACHANNEL;
   _string2EnumList[_uuidL2CAP] = (int)UUID_L2CAP;
   _string2EnumList[_uuidServiceDiscoveryServerServiceClassID] = (int)UUID_SERVICEDISCOVERYSERVERSERVICECLASSID;
   _string2EnumList[_uuidBrowseGroupDescriptorServiceClassID] = (int)UUID_BROWSEGROUPDESCRIPTORSERVICECLASSID;
   _string2EnumList[_uuidSerialPort] = (int)UUID_SERIALPORT;
   _string2EnumList[_uuidLANAccessUsingPPP] = (int)UUID_LANACCESSUSINGPPP;
   _string2EnumList[_uuidDialupNetworking] = (int)UUID_DIALUPNETWORKING;
   _string2EnumList[_uuidIrMCSync] = (int)UUID_IRMCSYNC;
   _string2EnumList[_uuidOBEXObjectPush] = (int)UUID_OBEXOBJECTPUSH;
   _string2EnumList[_uuidOBEXFileTransfer] = (int)UUID_OBEXFILETRANSFER;
   _string2EnumList[_uuidIrMCSyncCommand] = (int)UUID_IRMCSYNCCOMMAND;
   _string2EnumList[_uuidHeadset] = (int)UUID_HEADSET;
   _string2EnumList[_uuidCordlessTelephony] = (int)UUID_CORDLESSTELEPHONY;
   _string2EnumList[_uuidAudioSource] = (int)UUID_AUDIOSOURCE;
   _string2EnumList[_uuidAudioSink] = (int)UUID_AUDIOSINK;
   _string2EnumList[_uuidA_V_RemoteControlTarget] = (int)UUID_A_V_REMOTECONTROLTARGET;
   _string2EnumList[_uuidAdvancedAudioDistribution] = (int)UUID_ADVANCEDAUDIODISTRIBUTION;
   _string2EnumList[_uuidA_V_RemoteControl] = (int)UUID_A_V_REMOTECONTROL;
   _string2EnumList[_uuidA_V_RemoteControlController] = (int)UUID_A_V_REMOTECONTROLCONTROLLER;
   _string2EnumList[_uuidIntercom] = (int)UUID_INTERCOM;
   _string2EnumList[_uuidFax] = (int)UUID_FAX;
   _string2EnumList[_uuidHeadset_Audio_Gateway] = (int)UUID_HEADSET_AUDIO_GATEWAY;
   _string2EnumList[_uuidWAP] = (int)UUID_WAP;
   _string2EnumList[_uuidWAP_CLIENT] = (int)UUID_WAP_CLIENT;
   _string2EnumList[_uuidPANU] = (int)UUID_PANU;
   _string2EnumList[_uuidNAP] = (int)UUID_NAP;
   _string2EnumList[_uuidGN] = (int)UUID_GN;
   _string2EnumList[_uuidDirectPrinting] = (int)UUID_DIRECTPRINTING;
   _string2EnumList[_uuidReferencePrinting] = (int)UUID_REFERENCEPRINTING;
   _string2EnumList[_uuidBasic_Imaging_Profile] = (int)UUID_BASIC_IMAGING_PROFILE;
   _string2EnumList[_uuidImagingResponder] = (int)UUID_IMAGINGRESPONDER;
   _string2EnumList[_uuidImagingAutomaticArchive] = (int)UUID_IMAGINGAUTOMATICARCHIVE;
   _string2EnumList[_uuidImagingReferencedObjects] = (int)UUID_IMAGINGREFERENCEDOBJECTS;
   _string2EnumList[_uuidHandsfree] = (int)UUID_HANDSFREE;
   _string2EnumList[_uuidHandsfreeAudioGateway] = (int)UUID_HANDSFREEAUDIOGATEWAY;
   _string2EnumList[_uuidDirectPrintingReferenceObjectsService] = (int)UUID_DIRECTPRINTINGREFERENCEOBJECTSSERVICE;
   _string2EnumList[_uuidReflectedUI] = (int)UUID_REFLECTEDUI;
   _string2EnumList[_uuidBasicPrinting] = (int)UUID_BASICPRINTING;
   _string2EnumList[_uuidPrintingStatus] = (int)UUID_PRINTINGSTATUS;
   _string2EnumList[_uuidHumanInterfaceDeviceService] = (int)UUID_HUMANINTERFACEDEVICESERVICE;
   _string2EnumList[_uuidHardcopyCableReplacement] = (int)UUID_HARDCOPYCABLEREPLACEMENT;
   _string2EnumList[_uuidHCR_Print] = (int)UUID_HCR_PRINT;
   _string2EnumList[_uuidHCR_Scan] = (int)UUID_HCR_SCAN;
   _string2EnumList[_uuidCommon_ISDN_Access] = (int)UUID_COMMON_ISDN_ACCESS;
   _string2EnumList[_uuidSIM_Access] = (int)UUID_SIM_ACCESS;
   _string2EnumList[_uuidPhonebook_Access_PCE] = (int)UUID_PHONEBOOK_ACCESS_PCE;
   _string2EnumList[_uuidPhonebook_Access_PSE] = (int)UUID_PHONEBOOK_ACCESS_PSE;
   _string2EnumList[_uuidPhonebook_Access] = (int)UUID_PHONEBOOK_ACCESS;
   _string2EnumList[_uuidHeadset_HS] = (int)UUID_HEADSET_HS;
   _string2EnumList[_uuidMessage_Access_Server] = (int)UUID_MESSAGE_ACCESS_SERVER;
   _string2EnumList[_uuidMessage_Notification_Server] = (int)UUID_MESSAGE_NOTIFICATION_SERVER;
   _string2EnumList[_uuidMessage_Access_Profile] = (int)UUID_MESSAGE_ACCESS_PROFILE;
   _string2EnumList[_uuidGNSS] = (int)UUID_GNSS;
   _string2EnumList[_uuidGNSS_Server] = (int)UUID_GNSS_SERVER;
   _string2EnumList[_uuid3D_Display] = (int)UUID_3D_DISPLAY;
   _string2EnumList[_uuid3D_Glasses] = (int)UUID_3D_GLASSES;
   _string2EnumList[_uuid3D_Synchronization] = (int)UUID_3D_SYNCHRONIZATION;
   _string2EnumList[_uuidMPS_Profile] = (int)UUID_MPS_PROFILE;
   _string2EnumList[_uuidMPS_SC] = (int)UUID_MPS_SC;
   _string2EnumList[_uuidCTN_Access_Service] = (int)UUID_CTN_ACCESS_SERVICE;
   _string2EnumList[_uuidCTN_Notification_Service] = (int)UUID_CTN_NOTIFICATION_SERVICE;
   _string2EnumList[_uuidCTN_Profile] = (int)UUID_CTN_PROFILE;
   _string2EnumList[_uuidPnPInformation] = (int)UUID_PNPINFORMATION;
   _string2EnumList[_uuidGenericNetworking] = (int)UUID_GENERICNETWORKING;
   _string2EnumList[_uuidGenericFileTransfer] = (int)UUID_GENERICFILETRANSFER;
   _string2EnumList[_uuidGenericAudio] = (int)UUID_GENERICAUDIO;
   _string2EnumList[_uuidGenericTelephony] = (int)UUID_GENERICTELEPHONY;
   _string2EnumList[_uuidUPNP_Service] = (int)UUID_UPNP_SERVICE;
   _string2EnumList[_uuidUPNP_IP_Service] = (int)UUID_UPNP_IP_SERVICE;
   _string2EnumList[_uuidESDP_UPNP_IP_PAN] = (int)UUID_ESDP_UPNP_IP_PAN;
   _string2EnumList[_uuidESDP_UPNP_IP_LAP] = (int)UUID_ESDP_UPNP_IP_LAP;
   _string2EnumList[_uuidESDP_UPNP_L2CAP] = (int)UUID_ESDP_UPNP_L2CAP;
   _string2EnumList[_uuidVideoSource] = (int)UUID_VIDEOSOURCE;
   _string2EnumList[_uuidVideoSink] = (int)UUID_VIDEOSINK;
   _string2EnumList[_uuidVideoDistribution] = (int)UUID_VIDEODISTRIBUTION;
   _string2EnumList[_uuidHDP] = (int)UUID_HDP;
   _string2EnumList[_uuidHDP_Source] = (int)UUID_HDP_SOURCE;
   _string2EnumList[_uuidHDP_Sink] = (int)UUID_HDP_SINK;
   _string2EnumList[_uuidSyncMLService] = (int)UUID_SYNCML;

   _enum2StringList[(int)UUID_BASE] = &_uuidBase;
   _enum2StringList[(int)UUID_SDP] = &_uuidSDP;
   _enum2StringList[(int)UUID_UDP] = &_uuidUDP;
   _enum2StringList[(int)UUID_RFCOMM] = &_uuidRFCOMM;
   _enum2StringList[(int)UUID_TCP] = &_uuidTCP;
   _enum2StringList[(int)UUID_TCS_BIN] = &_uuidTCS_BIN;
   _enum2StringList[(int)UUID_TCS_AT] = &_uuidTCS_AT;
   _enum2StringList[(int)UUID_ATT] = &_uuidATT;
   _enum2StringList[(int)UUID_OBEX] = &_uuidOBEX;
   _enum2StringList[(int)UUID_IP] = &_uuidIP;
   _enum2StringList[(int)UUID_FTP] = &_uuidFTP;
   _enum2StringList[(int)UUID_HTTP] = &_uuidHTTP;
   _enum2StringList[(int)UUID_WSP] = &_uuidWSP;
   _enum2StringList[(int)UUID_BNEP] = &_uuidBNEP;
   _enum2StringList[(int)UUID_UPNP] = &_uuidUPNP;
   _enum2StringList[(int)UUID_HIDP] = &_uuidHIDP;
   _enum2StringList[(int)UUID_HARDCOPYCONTROLCHANNEL] = &_uuidHardcopyControlChannel;
   _enum2StringList[(int)UUID_HARDCOPYDATACHANNEL] = &_uuidHardcopyDataChannel;
   _enum2StringList[(int)UUID_HARDCOPYNOTIFICATION] = &_uuidHardcopyNotification;
   _enum2StringList[(int)UUID_AVCTP] = &_uuidAVCTP;
   _enum2StringList[(int)UUID_AVDTP] = &_uuidAVDTP;
   _enum2StringList[(int)UUID_CMTP] = &_uuidCMTP;
   _enum2StringList[(int)UUID_MCAPCONTROLCHANNEL] = &_uuidMCAPControlChannel;
   _enum2StringList[(int)UUID_MCAPDATACHANNEL] = &_uuidMCAPDataChannel;
   _enum2StringList[(int)UUID_L2CAP] = &_uuidL2CAP;
   _enum2StringList[(int)UUID_SERVICEDISCOVERYSERVERSERVICECLASSID] = &_uuidServiceDiscoveryServerServiceClassID;
   _enum2StringList[(int)UUID_BROWSEGROUPDESCRIPTORSERVICECLASSID] = &_uuidBrowseGroupDescriptorServiceClassID;
   _enum2StringList[(int)UUID_SERIALPORT] = &_uuidSerialPort;
   _enum2StringList[(int)UUID_LANACCESSUSINGPPP] = &_uuidLANAccessUsingPPP;
   _enum2StringList[(int)UUID_DIALUPNETWORKING] = &_uuidDialupNetworking;
   _enum2StringList[(int)UUID_IRMCSYNC] = &_uuidIrMCSync;
   _enum2StringList[(int)UUID_OBEXOBJECTPUSH] = &_uuidOBEXObjectPush;
   _enum2StringList[(int)UUID_OBEXFILETRANSFER] = &_uuidOBEXFileTransfer;
   _enum2StringList[(int)UUID_IRMCSYNCCOMMAND] = &_uuidIrMCSyncCommand;
   _enum2StringList[(int)UUID_HEADSET] = &_uuidHeadset;
   _enum2StringList[(int)UUID_CORDLESSTELEPHONY] = &_uuidCordlessTelephony;
   _enum2StringList[(int)UUID_AUDIOSOURCE] = &_uuidAudioSource;
   _enum2StringList[(int)UUID_AUDIOSINK] = &_uuidAudioSink;
   _enum2StringList[(int)UUID_A_V_REMOTECONTROLTARGET] = &_uuidA_V_RemoteControlTarget;
   _enum2StringList[(int)UUID_ADVANCEDAUDIODISTRIBUTION] = &_uuidAdvancedAudioDistribution;
   _enum2StringList[(int)UUID_A_V_REMOTECONTROL] = &_uuidA_V_RemoteControl;
   _enum2StringList[(int)UUID_A_V_REMOTECONTROLCONTROLLER] = &_uuidA_V_RemoteControlController;
   _enum2StringList[(int)UUID_INTERCOM] = &_uuidIntercom;
   _enum2StringList[(int)UUID_FAX] = &_uuidFax;
   _enum2StringList[(int)UUID_HEADSET_AUDIO_GATEWAY] = &_uuidHeadset_Audio_Gateway;
   _enum2StringList[(int)UUID_WAP] = &_uuidWAP;
   _enum2StringList[(int)UUID_WAP_CLIENT] = &_uuidWAP_CLIENT;
   _enum2StringList[(int)UUID_PANU] = &_uuidPANU;
   _enum2StringList[(int)UUID_NAP] = &_uuidNAP;
   _enum2StringList[(int)UUID_GN] = &_uuidGN;
   _enum2StringList[(int)UUID_DIRECTPRINTING] = &_uuidDirectPrinting;
   _enum2StringList[(int)UUID_REFERENCEPRINTING] = &_uuidReferencePrinting;
   _enum2StringList[(int)UUID_BASIC_IMAGING_PROFILE] = &_uuidBasic_Imaging_Profile;
   _enum2StringList[(int)UUID_IMAGINGRESPONDER] = &_uuidImagingResponder;
   _enum2StringList[(int)UUID_IMAGINGAUTOMATICARCHIVE] = &_uuidImagingAutomaticArchive;
   _enum2StringList[(int)UUID_IMAGINGREFERENCEDOBJECTS] = &_uuidImagingReferencedObjects;
   _enum2StringList[(int)UUID_HANDSFREE] = &_uuidHandsfree;
   _enum2StringList[(int)UUID_HANDSFREEAUDIOGATEWAY] = &_uuidHandsfreeAudioGateway;
   _enum2StringList[(int)UUID_DIRECTPRINTINGREFERENCEOBJECTSSERVICE] = &_uuidDirectPrintingReferenceObjectsService;
   _enum2StringList[(int)UUID_REFLECTEDUI] = &_uuidReflectedUI;
   _enum2StringList[(int)UUID_BASICPRINTING] = &_uuidBasicPrinting;
   _enum2StringList[(int)UUID_PRINTINGSTATUS] = &_uuidPrintingStatus;
   _enum2StringList[(int)UUID_HUMANINTERFACEDEVICESERVICE] = &_uuidHumanInterfaceDeviceService;
   _enum2StringList[(int)UUID_HARDCOPYCABLEREPLACEMENT] = &_uuidHardcopyCableReplacement;
   _enum2StringList[(int)UUID_HCR_PRINT] = &_uuidHCR_Print;
   _enum2StringList[(int)UUID_HCR_SCAN] = &_uuidHCR_Scan;
   _enum2StringList[(int)UUID_COMMON_ISDN_ACCESS] = &_uuidCommon_ISDN_Access;
   _enum2StringList[(int)UUID_SIM_ACCESS] = &_uuidSIM_Access;
   _enum2StringList[(int)UUID_PHONEBOOK_ACCESS_PCE] = &_uuidPhonebook_Access_PCE;
   _enum2StringList[(int)UUID_PHONEBOOK_ACCESS_PSE] = &_uuidPhonebook_Access_PSE;
   _enum2StringList[(int)UUID_PHONEBOOK_ACCESS] = &_uuidPhonebook_Access;
   _enum2StringList[(int)UUID_HEADSET_HS] = &_uuidHeadset_HS;
   _enum2StringList[(int)UUID_MESSAGE_ACCESS_SERVER] = &_uuidMessage_Access_Server;
   _enum2StringList[(int)UUID_MESSAGE_NOTIFICATION_SERVER] = &_uuidMessage_Notification_Server;
   _enum2StringList[(int)UUID_MESSAGE_ACCESS_PROFILE] = &_uuidMessage_Access_Profile;
   _enum2StringList[(int)UUID_GNSS] = &_uuidGNSS;
   _enum2StringList[(int)UUID_GNSS_SERVER] = &_uuidGNSS_Server;
   _enum2StringList[(int)UUID_3D_DISPLAY] = &_uuid3D_Display;
   _enum2StringList[(int)UUID_3D_GLASSES] = &_uuid3D_Glasses;
   _enum2StringList[(int)UUID_3D_SYNCHRONIZATION] = &_uuid3D_Synchronization;
   _enum2StringList[(int)UUID_MPS_PROFILE] = &_uuidMPS_Profile;
   _enum2StringList[(int)UUID_MPS_SC] = &_uuidMPS_SC;
   _enum2StringList[(int)UUID_CTN_ACCESS_SERVICE] = &_uuidCTN_Access_Service;
   _enum2StringList[(int)UUID_CTN_NOTIFICATION_SERVICE] = &_uuidCTN_Notification_Service;
   _enum2StringList[(int)UUID_CTN_PROFILE] = &_uuidCTN_Profile;
   _enum2StringList[(int)UUID_PNPINFORMATION] = &_uuidPnPInformation;
   _enum2StringList[(int)UUID_GENERICNETWORKING] = &_uuidGenericNetworking;
   _enum2StringList[(int)UUID_GENERICFILETRANSFER] = &_uuidGenericFileTransfer;
   _enum2StringList[(int)UUID_GENERICAUDIO] = &_uuidGenericAudio;
   _enum2StringList[(int)UUID_GENERICTELEPHONY] = &_uuidGenericTelephony;
   _enum2StringList[(int)UUID_UPNP_SERVICE] = &_uuidUPNP_Service;
   _enum2StringList[(int)UUID_UPNP_IP_SERVICE] = &_uuidUPNP_IP_Service;
   _enum2StringList[(int)UUID_ESDP_UPNP_IP_PAN] = &_uuidESDP_UPNP_IP_PAN;
   _enum2StringList[(int)UUID_ESDP_UPNP_IP_LAP] = &_uuidESDP_UPNP_IP_LAP;
   _enum2StringList[(int)UUID_ESDP_UPNP_L2CAP] = &_uuidESDP_UPNP_L2CAP;
   _enum2StringList[(int)UUID_VIDEOSOURCE] = &_uuidVideoSource;
   _enum2StringList[(int)UUID_VIDEOSINK] = &_uuidVideoSink;
   _enum2StringList[(int)UUID_VIDEODISTRIBUTION] = &_uuidVideoDistribution;
   _enum2StringList[(int)UUID_HDP] = &_uuidHDP;
   _enum2StringList[(int)UUID_HDP_SOURCE] = &_uuidHDP_Source;
   _enum2StringList[(int)UUID_HDP_SINK] = &_uuidHDP_Sink;
   _enum2StringList[(int)UUID_SYNCML] = &_uuidSyncMLService;

   checkUuidList();

   _enum2TraceList[(int)UUID_BASE] = &_traceBASE;
   _enum2TraceList[(int)UUID_SDP] = &_traceSDP;
   _enum2TraceList[(int)UUID_UDP] = &_traceUDP;
   _enum2TraceList[(int)UUID_RFCOMM] = &_traceRFCOMM;
   _enum2TraceList[(int)UUID_TCP] = &_traceTCP;
   _enum2TraceList[(int)UUID_TCS_BIN] = &_traceTCS_BIN;
   _enum2TraceList[(int)UUID_TCS_AT] = &_traceTCS_AT;
   _enum2TraceList[(int)UUID_ATT] = &_traceATT;
   _enum2TraceList[(int)UUID_OBEX] = &_traceOBEX;
   _enum2TraceList[(int)UUID_IP] = &_traceIP;
   _enum2TraceList[(int)UUID_FTP] = &_traceFTP;
   _enum2TraceList[(int)UUID_HTTP] = &_traceHTTP;
   _enum2TraceList[(int)UUID_WSP] = &_traceWSP;
   _enum2TraceList[(int)UUID_BNEP] = &_traceBNEP;
   _enum2TraceList[(int)UUID_UPNP] = &_traceUPNP;
   _enum2TraceList[(int)UUID_HIDP] = &_traceHIDP;
   _enum2TraceList[(int)UUID_HARDCOPYCONTROLCHANNEL] = &_traceHardcopyControlChannel;
   _enum2TraceList[(int)UUID_HARDCOPYDATACHANNEL] = &_traceHardcopyDataChannel;
   _enum2TraceList[(int)UUID_HARDCOPYNOTIFICATION] = &_traceHardcopyNotification;
   _enum2TraceList[(int)UUID_AVCTP] = &_traceAVCTP;
   _enum2TraceList[(int)UUID_AVDTP] = &_traceAVDTP;
   _enum2TraceList[(int)UUID_CMTP] = &_traceCMTP;
   _enum2TraceList[(int)UUID_MCAPCONTROLCHANNEL] = &_traceMCAPControlChannel;
   _enum2TraceList[(int)UUID_MCAPDATACHANNEL] = &_traceMCAPDataChannel;
   _enum2TraceList[(int)UUID_L2CAP] = &_traceL2CAP;
   _enum2TraceList[(int)UUID_SERVICEDISCOVERYSERVERSERVICECLASSID] = &_traceServiceDiscoveryServerServiceClassID;
   _enum2TraceList[(int)UUID_BROWSEGROUPDESCRIPTORSERVICECLASSID] = &_traceBrowseGroupDescriptorServiceClassID;
   _enum2TraceList[(int)UUID_SERIALPORT] = &_traceSerialPort;
   _enum2TraceList[(int)UUID_LANACCESSUSINGPPP] = &_traceLANAccessUsingPPP;
   _enum2TraceList[(int)UUID_DIALUPNETWORKING] = &_traceDialupNetworking;
   _enum2TraceList[(int)UUID_IRMCSYNC] = &_traceIrMCSync;
   _enum2TraceList[(int)UUID_OBEXOBJECTPUSH] = &_traceOBEXObjectPush;
   _enum2TraceList[(int)UUID_OBEXFILETRANSFER] = &_traceOBEXFileTransfer;
   _enum2TraceList[(int)UUID_IRMCSYNCCOMMAND] = &_traceIrMCSyncCommand;
   _enum2TraceList[(int)UUID_HEADSET] = &_traceHeadset;
   _enum2TraceList[(int)UUID_CORDLESSTELEPHONY] = &_traceCordlessTelephony;
   _enum2TraceList[(int)UUID_AUDIOSOURCE] = &_traceAudioSource;
   _enum2TraceList[(int)UUID_AUDIOSINK] = &_traceAudioSink;
   _enum2TraceList[(int)UUID_A_V_REMOTECONTROLTARGET] = &_traceA_V_RemoteControlTarget;
   _enum2TraceList[(int)UUID_ADVANCEDAUDIODISTRIBUTION] = &_traceAdvancedAudioDistribution;
   _enum2TraceList[(int)UUID_A_V_REMOTECONTROL] = &_traceA_V_RemoteControl;
   _enum2TraceList[(int)UUID_A_V_REMOTECONTROLCONTROLLER] = &_traceA_V_RemoteControlController;
   _enum2TraceList[(int)UUID_INTERCOM] = &_traceIntercom;
   _enum2TraceList[(int)UUID_FAX] = &_traceFax;
   _enum2TraceList[(int)UUID_HEADSET_AUDIO_GATEWAY] = &_traceHeadset_Audio_Gateway;
   _enum2TraceList[(int)UUID_WAP] = &_traceWAP;
   _enum2TraceList[(int)UUID_WAP_CLIENT] = &_traceWAP_CLIENT;
   _enum2TraceList[(int)UUID_PANU] = &_tracePANU;
   _enum2TraceList[(int)UUID_NAP] = &_traceNAP;
   _enum2TraceList[(int)UUID_GN] = &_traceGN;
   _enum2TraceList[(int)UUID_DIRECTPRINTING] = &_traceDirectPrinting;
   _enum2TraceList[(int)UUID_REFERENCEPRINTING] = &_traceReferencePrinting;
   _enum2TraceList[(int)UUID_BASIC_IMAGING_PROFILE] = &_traceBasic_Imaging_Profile;
   _enum2TraceList[(int)UUID_IMAGINGRESPONDER] = &_traceImagingResponder;
   _enum2TraceList[(int)UUID_IMAGINGAUTOMATICARCHIVE] = &_traceImagingAutomaticArchive;
   _enum2TraceList[(int)UUID_IMAGINGREFERENCEDOBJECTS] = &_traceImagingReferencedObjects;
   _enum2TraceList[(int)UUID_HANDSFREE] = &_traceHandsfree;
   _enum2TraceList[(int)UUID_HANDSFREEAUDIOGATEWAY] = &_traceHandsfreeAudioGateway;
   _enum2TraceList[(int)UUID_DIRECTPRINTINGREFERENCEOBJECTSSERVICE] = &_traceDirectPrintingReferenceObjectsService;
   _enum2TraceList[(int)UUID_REFLECTEDUI] = &_traceReflectedUI;
   _enum2TraceList[(int)UUID_BASICPRINTING] = &_traceBasicPrinting;
   _enum2TraceList[(int)UUID_PRINTINGSTATUS] = &_tracePrintingStatus;
   _enum2TraceList[(int)UUID_HUMANINTERFACEDEVICESERVICE] = &_traceHumanInterfaceDeviceService;
   _enum2TraceList[(int)UUID_HARDCOPYCABLEREPLACEMENT] = &_traceHardcopyCableReplacement;
   _enum2TraceList[(int)UUID_HCR_PRINT] = &_traceHCR_Print;
   _enum2TraceList[(int)UUID_HCR_SCAN] = &_traceHCR_Scan;
   _enum2TraceList[(int)UUID_COMMON_ISDN_ACCESS] = &_traceCommon_ISDN_Access;
   _enum2TraceList[(int)UUID_SIM_ACCESS] = &_traceSIM_Access;
   _enum2TraceList[(int)UUID_PHONEBOOK_ACCESS_PCE] = &_tracePhonebook_Access_PCE;
   _enum2TraceList[(int)UUID_PHONEBOOK_ACCESS_PSE] = &_tracePhonebook_Access_PSE;
   _enum2TraceList[(int)UUID_PHONEBOOK_ACCESS] = &_tracePhonebook_Access;
   _enum2TraceList[(int)UUID_HEADSET_HS] = &_traceHeadset_HS;
   _enum2TraceList[(int)UUID_MESSAGE_ACCESS_SERVER] = &_traceMessage_Access_Server;
   _enum2TraceList[(int)UUID_MESSAGE_NOTIFICATION_SERVER] = &_traceMessage_Notification_Server;
   _enum2TraceList[(int)UUID_MESSAGE_ACCESS_PROFILE] = &_traceMessage_Access_Profile;
   _enum2TraceList[(int)UUID_GNSS] = &_traceGNSS;
   _enum2TraceList[(int)UUID_GNSS_SERVER] = &_traceGNSS_Server;
   _enum2TraceList[(int)UUID_3D_DISPLAY] = &_trace3D_Display;
   _enum2TraceList[(int)UUID_3D_GLASSES] = &_trace3D_Glasses;
   _enum2TraceList[(int)UUID_3D_SYNCHRONIZATION] = &_trace3D_Synchronization;
   _enum2TraceList[(int)UUID_MPS_PROFILE] = &_traceMPS_Profile;
   _enum2TraceList[(int)UUID_MPS_SC] = &_traceMPS_SC;
   _enum2TraceList[(int)UUID_CTN_ACCESS_SERVICE] = &_traceCTN_Access_Service;
   _enum2TraceList[(int)UUID_CTN_NOTIFICATION_SERVICE] = &_traceCTN_Notification_Service;
   _enum2TraceList[(int)UUID_CTN_PROFILE] = &_traceCTN_Profile;
   _enum2TraceList[(int)UUID_PNPINFORMATION] = &_tracePnPInformation;
   _enum2TraceList[(int)UUID_GENERICNETWORKING] = &_traceGenericNetworking;
   _enum2TraceList[(int)UUID_GENERICFILETRANSFER] = &_traceGenericFileTransfer;
   _enum2TraceList[(int)UUID_GENERICAUDIO] = &_traceGenericAudio;
   _enum2TraceList[(int)UUID_GENERICTELEPHONY] = &_traceGenericTelephony;
   _enum2TraceList[(int)UUID_UPNP_SERVICE] = &_traceUPNP_Service;
   _enum2TraceList[(int)UUID_UPNP_IP_SERVICE] = &_traceUPNP_IP_Service;
   _enum2TraceList[(int)UUID_ESDP_UPNP_IP_PAN] = &_traceESDP_UPNP_IP_PAN;
   _enum2TraceList[(int)UUID_ESDP_UPNP_IP_LAP] = &_traceESDP_UPNP_IP_LAP;
   _enum2TraceList[(int)UUID_ESDP_UPNP_L2CAP] = &_traceESDP_UPNP_L2CAP;
   _enum2TraceList[(int)UUID_VIDEOSOURCE] = &_traceVideoSource;
   _enum2TraceList[(int)UUID_VIDEOSINK] = &_traceVideoSink;
   _enum2TraceList[(int)UUID_VIDEODISTRIBUTION] = &_traceVideoDistribution;
   _enum2TraceList[(int)UUID_HDP] = &_traceHDP;
   _enum2TraceList[(int)UUID_HDP_SOURCE] = &_traceHDP_Source;
   _enum2TraceList[(int)UUID_HDP_SINK] = &_traceHDP_Sink;
   _enum2TraceList[(int)UUID_SYNCML] = &_traceSYNCML;
}

EvolutionGeniviDbusUuids::~EvolutionGeniviDbusUuids()
{
   _enum2TraceList.clear();
}

UUID EvolutionGeniviDbusUuids::getUuid2Enum(const ::std::string& value) const
{
   // not sure if all phones are sending data in lower cases; therefore we will convert to lower cases before executing conversion
   ::std::string working(value);
   ::fw::convertString2LowerCase(working);

   if(false == isValidUuid(working, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_ERRMEM((" #CONN: CcDbusIf: getUuid2Enum(): proxy=UUIDs input=%s", working.c_str()));
      return UUID_UNKNOWN;
   }

   return (UUID)convertString2Enum(working);
}

const ::std::string& EvolutionGeniviDbusUuids::getUuid2String(const UUID value) const
{
   return convertEnum2String((const int)value);
}

const ::std::string& EvolutionGeniviDbusUuids::getUuid2StringTrace(const UUID value) const
{
   if(UUID_UNKNOWN <= value)
   {
      return _unknownString;
   }

   const ::std::string& result = convertEnum2String(_enum2TraceList, (const int)value, false);

   if(_dummyString == result)
   {
      return _unknownString;
   }

   return result;
}

bool EvolutionGeniviDbusUuids::isValidUuid(const ::std::string& value, const bool ignoreCaseSensitive /*= true*/) const
{
   ::std::string working(value);

   if(true == ignoreCaseSensitive)
   {
      ::fw::convertString2LowerCase(working);
   }

   // use _uuidBase for check (ensure that this one is correct)
   if(_uuidBase.size() != working.size())
   {
      // NOK
      return false;
   }

   for(size_t i = 0; i < working.size(); i++)
   {
      if('-' == working[i])
      {
         if(_uuidBase[i] == working[i])
         {
            // OK
         }
         else
         {
            // NOK
            return false;
         }
      }
      else if(('0' <= working[i]) && (working[i] <= '9'))
      {
         if(('0' <= _uuidBase[i]) && (_uuidBase[i] <= '9'))
         {
            // OK
         }
         else if(('a' <= _uuidBase[i]) && (_uuidBase[i] <= 'f'))
         {
            // OK
         }
         else
         {
            // NOK
            return false;
         }
      }
      else if(('a' <= working[i]) && (working[i] <= 'f'))
      {
         if(('0' <= _uuidBase[i]) && (_uuidBase[i] <= '9'))
         {
            // OK
         }
         else if(('a' <= _uuidBase[i]) && (_uuidBase[i] <= 'f'))
         {
            // OK
         }
         else
         {
            // NOK
            return false;
         }
      }
      else
      {
         // NOK
         return false;
      }
   }

   return true;
}

bool EvolutionGeniviDbusUuids::isShortUuid(UUID& uuid, const ::std::string& value) const
{
   ::std::string working(value);
   ::fw::convertString2LowerCase(working);

   // isValidUuid() shall be called before using this function
   if(false == isValidUuid(working, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   UUID shortUuid = (UUID)convertString2Enum(working, false);

   // ETG_TRACE_FATAL((" shortUuid=%d working=%s", shortUuid, working.c_str()));

   if(UUID_BASE == shortUuid)
   {
#if 0
      // given UUID not found => UUID is not in internal list => check for short UUID
      if(true == isUnknownShortUuid(working))
      {
         // short UUID but not in internal list
         uuid = UUID_UNKNOWN;
         return true;
      }
      else
#endif
      {
         // no short UUID
         return false;
      }
   }
   else
   {
      // short UUID
      uuid = shortUuid;
      return true;
   }
}

void EvolutionGeniviDbusUuids::checkUuidList(void) const
{
   /*
    * check that all UUIDs have lower case characters
    */
   for(::std::map< std::string, int >::const_iterator it = _string2EnumList.begin(); it != _string2EnumList.end(); ++it)
   {
      if(false == isValidUuid(it->first, false))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         ETG_TRACE_FATAL((" checkUuidList: UUID=%s has upper cases", it->first.c_str()));
      }
   }
}

bool EvolutionGeniviDbusUuids::isUnknownShortUuid(const ::std::string& value) const
{
   // BASE UUID:    00000000-0000-1000-8000-00805f9b34fb
   // unknown UUID: 00001101-0000-1000-8000-00805f9b34fb

   // given value was checked before but we have make lint happy

   const size_t pos = _uuid32.size(); // start after first 32 bits (covers "short UUIDs" (uuid16 and uuid32))
   if((_uuidBase.size() > pos) && (value.size() > pos))
   {
      // ETG_TRACE_FATAL((" _uuidBase=%s", _uuidBase.c_str()));
      // ETG_TRACE_FATAL((" value    =%s", value.c_str()));
      // ETG_TRACE_FATAL((" compare  =%d", _uuidBase.compare(pos, (_uuidBase.size() - pos), value, pos, (value.size() - pos))));

      return (0 == _uuidBase.compare(pos, (_uuidBase.size() - pos), value, pos, (value.size() - pos)));
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }
}

} //evolution
} //ccdbusif
