/**
 * @file EvolutionGeniviDbusServiceProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for Service.
 */

#include "EvolutionGeniviDbusServiceProxyIf.h"
#include "IEvolutionGeniviDbusServiceCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::Service;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusServiceProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusServiceProxyIf::EvolutionGeniviDbusServiceProxyIf()
{
   // _serviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusServiceProxyIf::EvolutionGeniviDbusServiceProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _serviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusServiceProxyIf::~EvolutionGeniviDbusServiceProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusServiceProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusServiceProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Service"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _serviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusServiceProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Service"));

   for(unsigned int i = 0; i < _serviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ServiceProxy >& serviceProxy = _serviceProxy[i];
      if(NULL != serviceProxy.get())
      {
         serviceProxy->sendDeregisterAll();
      }
   }

   _serviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusServiceProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Service: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _serviceProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< ServiceProxy > serviceProxy = ServiceProxy::createProxy("evoServicePort", parser.getInterface2BusName(evolution::IF_SERVICE), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_SERVICE)), *this);
         _serviceProxy.addProxyInstance(objPath, serviceProxy);
      }
   }
}

void EvolutionGeniviDbusServiceProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Service: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< ServiceProxy > serviceProxy;
   if(true == _serviceProxy.isProxyAvailable(serviceProxy, objPath))
   {
      serviceProxy->sendDeregisterAll();
   }
   _serviceProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusServiceProxyIf::setCallbackIf(IEvolutionGeniviDbusServiceCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusServiceProxyIf::serviceCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusServiceProxyIf::serviceRemoved(const ::std::string& objPath, const bool destroyDirectly /*= false*/)
{
   // destroy runtime proxy
   if (true == destroyDirectly)
   {
      // Called within context of ASF component thread, we can directly destroy the proxy.
      destroyRuntimeProxy(objPath);
   }
   else
   {
      if ((NULL != _callbackIf) && (NULL != _controllerClient))
      {
         _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
      }
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusServiceProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Service"));

   for(unsigned int i = 0; i < _serviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ServiceProxy >& serviceProxy = _serviceProxy[i];
      if((NULL != serviceProxy.get()) && (proxy == serviceProxy))
      {
         if(NULL != _callbackIf)
         {
            serviceProxy->sendRemoteUUIDRegister(*_callbackIf);
            serviceProxy->sendStateRegister(*_callbackIf);
            serviceProxy->sendDeviceRegister(*_callbackIf);
            serviceProxy->sendVersionRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), serviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusServiceProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Service"));

   for(unsigned int i = 0; i < _serviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ServiceProxy >& serviceProxy = _serviceProxy[i];
      if((NULL != serviceProxy.get()) && (proxy == serviceProxy))
      {
         serviceProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), serviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEvolutionGeniviDbusServiceProxyIf implementation --- start
act_t EvolutionGeniviDbusServiceProxyIf::sendConnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendConnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusServiceProxyIf::sendDisconnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDisconnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusServiceProxyIf::sendRemoteUUIDGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendRemoteUUIDGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusServiceProxyIf::sendStateGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendStateGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusServiceProxyIf::sendDeviceGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDeviceGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusServiceProxyIf::sendVersionGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< ServiceProxy > proxy;
   if(true == _serviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendVersionGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusServiceProxyIf implementation --- end

} //ccdbusif
