/**
 * @file EvolutionGeniviDbusSerialProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS proxy for Serial.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for Serial.
 */

#include "EvolutionGeniviDbusSerialProxyIf.h"
#include "EvolutionGeniviDbusParser.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

using namespace ::asf::core;
using namespace ::org::bluez::Serial;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusSerialProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace evolution {

EvolutionGeniviDbusSerialProxyIf::EvolutionGeniviDbusSerialProxyIf(ICcDbusIfControllerClient* client) :
DbusBaseProxyIf< IEvolutionGeniviDbusSerialCallbackIf, SerialProxy, IEvolutionGeniviDbusSerialTestProxyIf >(client)
{
}

EvolutionGeniviDbusSerialProxyIf::EvolutionGeniviDbusSerialProxyIf(ICcDbusIfControllerClient* client, IEvolutionGeniviDbusSerialTestProxyIf* testProxyIf) :
DbusBaseProxyIf< IEvolutionGeniviDbusSerialCallbackIf, SerialProxy, IEvolutionGeniviDbusSerialTestProxyIf >(client, testProxyIf)
{
}

EvolutionGeniviDbusSerialProxyIf::~EvolutionGeniviDbusSerialProxyIf()
{
}

void EvolutionGeniviDbusSerialProxyIf::onAvailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< SerialProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< SerialProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            (void)myProxy->sendSppDataIndRegister(*this);
            (void)myProxy->sendFlowCtrlSendRegister(*this);
            (void)myProxy->sendDeviceFilePathRegister(*this);
            (void)myProxy->sendFlowCtrlRecvRegister(*this);

            internalOnAvailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onUnavailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< SerialProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< SerialProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            myProxy->sendDeregisterAll();

            internalOnUnavailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::handleSetCallback(IEvolutionGeniviDbusSerialCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   storeCallback(callbackIf, enableProxy, callbackId); // independent of path, name, type => update status to all
}

void EvolutionGeniviDbusSerialProxyIf::handleCreateProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalCreateProxy(callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::handleDestroyProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalDestroyProxy(callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::handleDestroyAllProxies(void)
{
   internalDestroyAllProxies();
}

void EvolutionGeniviDbusSerialProxyIf::handleDestroyAllRuntimeProxies(void)
{
   internalDestroyAllRuntimeProxies();
}

IDestroyAllProxies* EvolutionGeniviDbusSerialProxyIf::getDestroyer(void)
{
   return this;
}

void EvolutionGeniviDbusSerialProxyIf::setCallbackIf(IEvolutionGeniviDbusSerialCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processSetCallback(this, callbackIf, enableProxy, callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::createProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processCreateProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::destroyProxyIf(const unsigned int callbackId, const bool destroyDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processDestroyProxy(this, destroyDirectly, callbackId, objPath, busName, busType);
}

act_t EvolutionGeniviDbusSerialProxyIf::sendSendDataRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::vector< uint8 >& data)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendSendDataRequest(*this, objPath, data);
      }
   }
   else
   {
      ::boost::shared_ptr< SerialProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendSendDataRequest(*this, data);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusSerialProxyIf::sendFlowCtrlRecvGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendFlowCtrlRecvGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< SerialProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendFlowCtrlRecvGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusSerialProxyIf::sendFlowCtrlRecvSet(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& flowCtrlRecv)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         _testProxyIf->sendFlowCtrlRecvSet(*this, objPath, flowCtrlRecv);
      }
   }
   else
   {
      ::boost::shared_ptr< SerialProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         proxy->sendFlowCtrlRecvSet(flowCtrlRecv);
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onSendDataError(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< SendDataError >& error)
{
   onSendDataErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onSendDataResponse(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< SendDataResponse >& response)
{
   onSendDataResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onSppDataIndError(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< SppDataIndError >& error)
{
   onSppDataIndErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onSppDataIndSignal(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< SppDataIndSignal >& signal)
{
   onSppDataIndSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlSendError(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< FlowCtrlSendError >& error)
{
   onFlowCtrlSendErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlSendSignal(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< FlowCtrlSendSignal >& signal)
{
   onFlowCtrlSendSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onDeviceFilePathError(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< DeviceFilePathError >& error)
{
   onDeviceFilePathErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onDeviceFilePathSignal(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< DeviceFilePathSignal >& signal)
{
   onDeviceFilePathSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlRecvError(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< FlowCtrlRecvError >& error)
{
   onFlowCtrlRecvErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlRecvUpdate(const ::boost::shared_ptr< SerialProxy >& proxy, const ::boost::shared_ptr< FlowCtrlRecvUpdate >& update)
{
   onFlowCtrlRecvUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusSerialProxyIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnAvailable(available, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnUnavailable(available, objPath, busName, busType);
}

void EvolutionGeniviDbusSerialProxyIf::onSendDataErrorCb(const ::boost::shared_ptr< SendDataError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSendDataErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusSerialProxyIf::onSendDataResponseCb(const ::boost::shared_ptr< SendDataResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSendDataResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusSerialProxyIf::onSppDataIndErrorCb(const ::boost::shared_ptr< SppDataIndError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onSppDataIndErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onSppDataIndErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onSppDataIndSignalCb(const ::boost::shared_ptr< SppDataIndSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onSppDataIndSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onSppDataIndSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlSendErrorCb(const ::boost::shared_ptr< FlowCtrlSendError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onFlowCtrlSendErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFlowCtrlSendErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlSendSignalCb(const ::boost::shared_ptr< FlowCtrlSendSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onFlowCtrlSendSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFlowCtrlSendSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onDeviceFilePathErrorCb(const ::boost::shared_ptr< DeviceFilePathError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDeviceFilePathErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceFilePathErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onDeviceFilePathSignalCb(const ::boost::shared_ptr< DeviceFilePathSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onDeviceFilePathSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceFilePathSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlRecvErrorCb(const ::boost::shared_ptr< FlowCtrlRecvError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onFlowCtrlRecvErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFlowCtrlRecvErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::onFlowCtrlRecvUpdateCb(const ::boost::shared_ptr< FlowCtrlRecvUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusSerialCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onFlowCtrlRecvUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFlowCtrlRecvUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusSerialProxyIf::internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   evolution::EvolutionGeniviDbusParser parser;
   const ::std::string& port(parser.getPort(evolution::IF_SERIAL));
   const ::std::string& path(objPath);
   const ::std::string& name(parser.getBusName(evolution::IF_SERIAL));
   const DbusBusType type(parser.getBusType(evolution::IF_SERIAL));

   // check if proxy is available
   if(false == isProxyAvailable(objPath)) // independent of name, type
   {
      ETG_TRACE_USR3((" internalCreateProxy(): Serial: callbackId=%u busType=%d busName=%20s objPath=%s (create)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // create proxy
      if(0 != _testProxyIf)
      {
         createTestProxy(port, path, name, type, *this);
         addTestProxy(objPath); // independent of name, type
      }
      else
      {
         ::boost::shared_ptr< SerialProxy > proxy = createProxy(port, path, name, type, *this);
         addProxy(proxy, objPath); // independent of name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalCreateProxy(): Serial: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // proxy creation was already started; either service is available or creation is ongoing
      updateCurrentAvailableStatus(callbackId, getCreationState(objPath) /* independent of name, type */, path, name, type);
   }
}

void EvolutionGeniviDbusSerialProxyIf::internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy
   // HINT: sendDeregisterAll() is not provided by all proxies

   // check if proxy is available
   if(true == isProxyAvailable(objPath)) // independent of name, type
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): Serial: callbackId=%u busType=%d busName=%20s objPath=%s (destroy)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // destroy proxy
      if(0 != _testProxyIf)
      {
         evolution::EvolutionGeniviDbusParser parser;
         const ::std::string& path(objPath);
         const ::std::string& name(parser.getBusName(evolution::IF_SERIAL));
         const DbusBusType type(parser.getBusType(evolution::IF_SERIAL));

         destroyTestProxy(path, name, type);
         removeTestProxy(objPath); // independent of name, type
      }
      else
      {
         ::boost::shared_ptr< SerialProxy >* proxyPtr(getProxy(objPath)); // independent of name, type
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< SerialProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
         removeProxy(objPath); // independent of name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): Serial: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));
   }

   // NOTE: check following: second callback is registered, anything to do?
}

void EvolutionGeniviDbusSerialProxyIf::internalDestroyAllProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllProxies(): Serial"));

   // destroy all proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< SerialProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< SerialProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // remove callback information
   removeCallbacks();
}

void EvolutionGeniviDbusSerialProxyIf::internalDestroyAllRuntimeProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllRuntimeProxies(): Serial"));

   // destroy all runtime proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< SerialProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< SerialProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // do not remove callback information
}

void EvolutionGeniviDbusSerialProxyIf::internalOnAvailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnAvailable(): Serial"));

   // inform all matching callback handler
   setCreationState(available, objPath); // independent of name, type
   updateAvailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

void EvolutionGeniviDbusSerialProxyIf::internalOnUnavailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnUnavailable(): Serial"));

   // inform all matching callback handler
   setCreationState(available, objPath); // independent of name, type
   updateUnavailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

} //evolution
} //ccdbusif
