/**
 * @file EvolutionGeniviDbusObexTransferProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for ObexTransfer.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for ObexTransfer.
 */

#include "EvolutionGeniviDbusObexTransferProxyIf.h"
#include "IEvolutionGeniviDbusObexTransferCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::obex::Transfer;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusObexTransferProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusObexTransferProxyIf::EvolutionGeniviDbusObexTransferProxyIf()
{
   // _obexTransferProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexTransferProxyIf::EvolutionGeniviDbusObexTransferProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _obexTransferProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexTransferProxyIf::~EvolutionGeniviDbusObexTransferProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusObexTransferProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusObexTransferProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Transfer"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _obexTransferProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusObexTransferProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Transfer"));

   for(unsigned int i = 0; i < _obexTransferProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TransferProxy >& obexTransferProxy = _obexTransferProxy[i];
      if(NULL != obexTransferProxy.get())
      {
         obexTransferProxy->sendDeregisterAll();
      }
   }

   _obexTransferProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusObexTransferProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Transfer: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _obexTransferProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< TransferProxy > obexTransferProxy = TransferProxy::createProxy("evoObexTransferPort", parser.getInterface2BusName(evolution::IF_TRANSFER), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_TRANSFER)), *this);
         _obexTransferProxy.addProxyInstance(objPath, obexTransferProxy);
      }
   }
}

void EvolutionGeniviDbusObexTransferProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Transfer: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< TransferProxy > obexTransferProxy;
   if(true == _obexTransferProxy.isProxyAvailable(obexTransferProxy, objPath))
   {
      obexTransferProxy->sendDeregisterAll();
   }
   _obexTransferProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusObexTransferProxyIf::setCallbackIf(IEvolutionGeniviDbusObexTransferCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusObexTransferProxyIf::transferCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusObexTransferProxyIf::transferRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusObexTransferProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Transfer"));

   for(unsigned int i = 0; i < _obexTransferProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TransferProxy >& obexTransferProxy = _obexTransferProxy[i];
      if((NULL != obexTransferProxy.get()) && (proxy == obexTransferProxy))
      {
         if(NULL != _callbackIf)
         {
            obexTransferProxy->sendStatusRegister(*_callbackIf);
            obexTransferProxy->sendSessionRegister(*_callbackIf);
            obexTransferProxy->sendNameRegister(*_callbackIf);
            obexTransferProxy->sendTypeRegister(*_callbackIf);
            obexTransferProxy->sendTimeRegister(*_callbackIf);
            obexTransferProxy->sendSizeRegister(*_callbackIf);
            obexTransferProxy->sendTransferredRegister(*_callbackIf);
            obexTransferProxy->sendFilenameRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), obexTransferProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusObexTransferProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Transfer"));

   for(unsigned int i = 0; i < _obexTransferProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TransferProxy >& obexTransferProxy = _obexTransferProxy[i];
      if((NULL != obexTransferProxy.get()) && (proxy == obexTransferProxy))
      {
         obexTransferProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), obexTransferProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEvolutionGeniviDbusObexTransferProxyIf implementation --- start
act_t EvolutionGeniviDbusObexTransferProxyIf::sendCancelRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendCancelRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendStatusGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendStatusGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendSessionGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendSessionGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendNameGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendNameGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendTypeGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendTypeGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendTimeGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendTimeGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendSizeGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendSizeGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendTransferredGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendTransferredGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexTransferProxyIf::sendFilenameGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< TransferProxy > proxy;
   if(true == _obexTransferProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendFilenameGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusObexTransferProxyIf implementation --- end

} //ccdbusif
