/**
 * @file EvolutionGeniviDbusLEDeviceProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS proxy for LEDevice.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for LEDevice.
 */

#include "EvolutionGeniviDbusLEDeviceProxyIf.h"
#include "IEvolutionGeniviDbusLEDeviceCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::LE_Device1;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusLEDeviceProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusLEDeviceProxyIf::EvolutionGeniviDbusLEDeviceProxyIf()
{
   // _lEDeviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusLEDeviceProxyIf::EvolutionGeniviDbusLEDeviceProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _lEDeviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusLEDeviceProxyIf::~EvolutionGeniviDbusLEDeviceProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusLEDeviceProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusLEDeviceProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): LE_Device1"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _lEDeviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusLEDeviceProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): LE_Device1"));

   for(unsigned int i = 0; i < _lEDeviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< LE_Device1Proxy >& lEDeviceProxy = _lEDeviceProxy[i];
      if(NULL != lEDeviceProxy.get())
      {
         lEDeviceProxy->sendDeregisterAll();
      }
   }

   _lEDeviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusLEDeviceProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): LE_Device1: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _lEDeviceProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< LE_Device1Proxy > lEDeviceProxy = LE_Device1Proxy::createProxy("evoLEDevicePort", parser.getInterface2BusName(evolution::IF_LE_DEVICE), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_LE_DEVICE)), *this);
         _lEDeviceProxy.addProxyInstance(objPath, lEDeviceProxy);
      }
   }
}

void EvolutionGeniviDbusLEDeviceProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): LE_Device1: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< LE_Device1Proxy > lEDeviceProxy;
   if(true == _lEDeviceProxy.isProxyAvailable(lEDeviceProxy, objPath))
   {
      lEDeviceProxy->sendDeregisterAll();
   }
   _lEDeviceProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusLEDeviceProxyIf::setCallbackIf(IEvolutionGeniviDbusLEDeviceCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusLEDeviceProxyIf::deviceCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusLEDeviceProxyIf::deviceRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusLEDeviceProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): LE_Device1"));

   for(unsigned int i = 0; i < _lEDeviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< LE_Device1Proxy >& lEDeviceProxy = _lEDeviceProxy[i];
      if((NULL != lEDeviceProxy.get()) && (proxy == lEDeviceProxy))
      {
         if(NULL != _callbackIf)
         {
            lEDeviceProxy->sendDisconnectionCompRegister(*_callbackIf);
            lEDeviceProxy->sendAddressTypeRegister(*_callbackIf);
            lEDeviceProxy->sendAddressRegister(*_callbackIf);
            lEDeviceProxy->sendConnectedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), lEDeviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusLEDeviceProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): LE_Device1"));

   for(unsigned int i = 0; i < _lEDeviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< LE_Device1Proxy >& lEDeviceProxy = _lEDeviceProxy[i];
      if((NULL != lEDeviceProxy.get()) && (proxy == lEDeviceProxy))
      {
         lEDeviceProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), lEDeviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusLEDeviceProxyIf implementation --- start
act_t EvolutionGeniviDbusLEDeviceProxyIf::sendDisconnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< LE_Device1Proxy > proxy;
   if(true == _lEDeviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDisconnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusLEDeviceProxyIf::sendAddressTypeGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< LE_Device1Proxy > proxy;
   if(true == _lEDeviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAddressTypeGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusLEDeviceProxyIf::sendAddressGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< LE_Device1Proxy > proxy;
   if(true == _lEDeviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAddressGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusLEDeviceProxyIf::sendConnectedGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< LE_Device1Proxy > proxy;
   if(true == _lEDeviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendConnectedGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusLEDeviceProxyIf implementation --- end

} //ccdbusif
