/**
 * @file EvolutionGeniviDbusInterfaces.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Interfaces.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Interfaces.
 */

#include "EvolutionGeniviDbusInterfaces.h"
#include "FwAssert.h"

namespace ccdbusif {
namespace evolution {

EvolutionGeniviDbusInterfaces::EvolutionGeniviDbusInterfaces() :
   _serviceStateDisconnected("Disconnected"),
   _serviceStateConnecting("Connecting"),
   _serviceStateConnected("Connected"),
   _serviceStateDisconnecting("Disconnecting"),
   _serviceStateError("Error")
{
}

EvolutionGeniviDbusInterfaces::~EvolutionGeniviDbusInterfaces()
{
}

void EvolutionGeniviDbusInterfaces::getInterfaces(::std::vector<const ::std::map< std::string, int >*>& interfaceList, const ::std::vector< int >& matchingInterfaces) const
{
   interfaceList.reserve(matchingInterfaces.size());

   for(size_t i = 0; i < matchingInterfaces.size(); i++)
   {
      switch((Interface)matchingInterfaces[i])
      {
         case IF_ADAPTER:
            interfaceList.push_back(_adapterProperties.getString2EnumListPtr());
            break;
         case IF_AGENT_MANAGER:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_DEVICE:
            interfaceList.push_back(_deviceProperties.getString2EnumListPtr());
            break;
         case IF_SERVICE:
            interfaceList.push_back(_serviceProperties.getString2EnumListPtr());
            break;
         case IF_TRACE:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_DM:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_SPP:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_DID:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_DUN:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_PAN:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_HANDSFREE:
            interfaceList.push_back(_hfpHandsfreeProperties.getString2EnumListPtr());
            break;
         case IF_MANAGER:
            // no properties
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case IF_MODEM:
            interfaceList.push_back(_hfpModemProperties.getString2EnumListPtr());
            break;
         case IF_VOICE_CALL:
            break;
         case IF_VOICE_CALL_MANAGER:
            break;
         case IF_NETWORK_REGISTRATION:
            break;
         case IF_NETWORK_OPERATOR:
            break;
         case IF_CALL_VOLUME:
            break;
         case IF_SIRI:
            break;
         case IF_AVP:
            break;
         case IF_MEDIA_FOLDER:
            break;
         case IF_MEDIA_ITEM:
            break;
         case IF_MEDIA_PLAYER:
            break;
         case IF_MEDIA_TRANSPORT:
            break;
         case IF_CLIENT:
            break;
         case IF_PHONEBOOK_ACCESS:
            break;
         case IF_SESSION:
            interfaceList.push_back(_obexSessionProperties.getString2EnumListPtr());
            break;
         case IF_TRANSFER:
            break;
         case IF_MESSAGE:
            break;
         case IF_MESSAGE_ACCESS:
            break;
         case IF_NETWORK:
            break;
         case IF_SERIAL:
            interfaceList.push_back(_serialProperties.getString2EnumListPtr());
            break;
         case IF_LE_ADAPTER:
            interfaceList.push_back(_leAdapterProperties.getString2EnumListPtr());
            break;
         case IF_LE_DEVICE:
            interfaceList.push_back(_leDeviceProperties.getString2EnumListPtr());
            break;
         case IF_LE_SEAMLESS_PAIRING:
            interfaceList.push_back(_leSeamlessPairingProperties.getString2EnumListPtr());
            break;
         case IF_UNKNOWN:
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
}

ServiceState EvolutionGeniviDbusInterfaces::convertString2ServiceState(const ::std::string& value) const
{
   ServiceState state = SERVICE_STATE_UNKNOWN;

   if(_serviceStateDisconnected == value)
   {
      state = SERVICE_STATE_DISCONNECTED;
   }
   else if(_serviceStateConnecting == value)
   {
      state = SERVICE_STATE_CONNECTING;
   }
   else if(_serviceStateConnected == value)
   {
      state = SERVICE_STATE_CONNECTED;
   }
   else if(_serviceStateDisconnecting == value)
   {
      state = SERVICE_STATE_DISCONNECTING;
   }
   else if(_serviceStateError == value)
   {
      state = SERVICE_STATE_ERROR;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return state;
}

} //evolution
} //ccdbusif
