/**
 * @file EvolutionGeniviDbusHfpVoiceCallProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for HfpVoiceCall.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for HfpVoiceCall.
 */

#include "EvolutionGeniviDbusHfpVoiceCallProxyIf.h"
#include "IEvolutionGeniviDbusHfpVoiceCallCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::ofono::VoiceCall;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpVoiceCallProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusHfpVoiceCallProxyIf::EvolutionGeniviDbusHfpVoiceCallProxyIf()
{
   // _hfpVoiceCallProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpVoiceCallProxyIf::EvolutionGeniviDbusHfpVoiceCallProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _hfpVoiceCallProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpVoiceCallProxyIf::~EvolutionGeniviDbusHfpVoiceCallProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): VoiceCall"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpVoiceCallProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): VoiceCall"));

   for(unsigned int i = 0; i < _hfpVoiceCallProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< VoiceCallProxy >& hfpVoiceCallProxy = _hfpVoiceCallProxy[i];
      if(NULL != hfpVoiceCallProxy.get())
      {
         hfpVoiceCallProxy->sendDeregisterAll();
      }
   }

   _hfpVoiceCallProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): VoiceCall: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _hfpVoiceCallProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< VoiceCallProxy > hfpVoiceCallProxy = VoiceCallProxy::createProxy("evoHfpVoiceCallPort", parser.getInterface2BusName(evolution::IF_VOICE_CALL), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_VOICE_CALL)), *this);
         _hfpVoiceCallProxy.addProxyInstance(objPath, hfpVoiceCallProxy);
      }
   }
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): VoiceCall: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< VoiceCallProxy > hfpVoiceCallProxy;
   if(true == _hfpVoiceCallProxy.isProxyAvailable(hfpVoiceCallProxy, objPath))
   {
      hfpVoiceCallProxy->sendDeregisterAll();
   }
   _hfpVoiceCallProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::setCallbackIf(IEvolutionGeniviDbusHfpVoiceCallCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::callCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::callRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusHfpVoiceCallProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): VoiceCall"));

   for(unsigned int i = 0; i < _hfpVoiceCallProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< VoiceCallProxy >& hfpVoiceCallProxy = _hfpVoiceCallProxy[i];
      if((NULL != hfpVoiceCallProxy.get()) && (proxy == hfpVoiceCallProxy))
      {
         if(NULL != _callbackIf)
         {
            hfpVoiceCallProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpVoiceCallProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpVoiceCallProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): VoiceCall"));

   for(unsigned int i = 0; i < _hfpVoiceCallProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< VoiceCallProxy >& hfpVoiceCallProxy = _hfpVoiceCallProxy[i];
      if((NULL != hfpVoiceCallProxy.get()) && (proxy == hfpVoiceCallProxy))
      {
         hfpVoiceCallProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpVoiceCallProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusHfpVoiceCallProxyIf implementation --- start
act_t EvolutionGeniviDbusHfpVoiceCallProxyIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< VoiceCallProxy > proxy;
   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpVoiceCallProxyIf::sendHangupRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< VoiceCallProxy > proxy;
   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendHangupRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpVoiceCallProxyIf::sendAnswerRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< VoiceCallProxy > proxy;
   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAnswerRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpVoiceCallProxyIf::sendHoldRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< VoiceCallProxy > proxy;
   if(true == _hfpVoiceCallProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendHoldRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusHfpVoiceCallProxyIf implementation --- end

} //ccdbusif
