/**
 * @file EvolutionGeniviDbusHfpSiriProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for HfpSiri.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for HfpSiri.
 */

#include "EvolutionGeniviDbusHfpSiriProxyIf.h"
#include "IEvolutionGeniviDbusHfpSiriCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::ofono::evo::Siri;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpSiriProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusHfpSiriProxyIf::EvolutionGeniviDbusHfpSiriProxyIf()
{
   // _hfpSiriProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpSiriProxyIf::EvolutionGeniviDbusHfpSiriProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _hfpSiriProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpSiriProxyIf::~EvolutionGeniviDbusHfpSiriProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusHfpSiriProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpSiriProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Siri"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpSiriProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpSiriProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Siri"));

   for(unsigned int i = 0; i < _hfpSiriProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SiriProxy >& hfpSiriProxy = _hfpSiriProxy[i];
      if(NULL != hfpSiriProxy.get())
      {
         hfpSiriProxy->sendDeregisterAll();
      }
   }

   _hfpSiriProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpSiriProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Siri: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _hfpSiriProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< SiriProxy > hfpSiriProxy = SiriProxy::createProxy("evoHfpSiri", parser.getInterface2BusName(evolution::IF_SIRI), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_SIRI)), *this);
         _hfpSiriProxy.addProxyInstance(objPath, hfpSiriProxy);
      }
   }
}

void EvolutionGeniviDbusHfpSiriProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Siri: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< SiriProxy > hfpSiriProxy;
   if(true == _hfpSiriProxy.isProxyAvailable(hfpSiriProxy, objPath))
   {
      hfpSiriProxy->sendDeregisterAll();
   }
   _hfpSiriProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpSiriProxyIf::setCallbackIf(IEvolutionGeniviDbusHfpSiriCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpSiriProxyIf::deviceCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusHfpSiriProxyIf::deviceRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusHfpSiriProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Siri"));

   for(unsigned int i = 0; i < _hfpSiriProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SiriProxy >& hfpSiriProxy = _hfpSiriProxy[i];
      if((NULL != hfpSiriProxy.get()) && (proxy == hfpSiriProxy))
      {
         if(NULL != _callbackIf)
         {
            hfpSiriProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpSiriProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpSiriProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Siri"));

   for(unsigned int i = 0; i < _hfpSiriProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SiriProxy >& hfpSiriProxy = _hfpSiriProxy[i];
      if((NULL != hfpSiriProxy.get()) && (proxy == hfpSiriProxy))
      {
         hfpSiriProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpSiriProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusHfpSiriProxyIf implementation --- start
act_t EvolutionGeniviDbusHfpSiriProxyIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< SiriProxy > proxy;
   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpSiriProxyIf::sendEnableRequest(const ::std::string& objPath, uint16 vendor_id, uint16 product_id, uint16 version, uint16 support_type)
{
   ::boost::shared_ptr< SiriProxy > proxy;
   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendEnableRequest(*_callbackIf, vendor_id, product_id, version, support_type);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpSiriProxyIf::sendSetNRRequest(const ::std::string& objPath, uint8 status)
{
   ::boost::shared_ptr< SiriProxy > proxy;
   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendSetNRRequest(*_callbackIf, status);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpSiriProxyIf::sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const bool value)
{
   ::boost::shared_ptr< SiriProxy > proxy;
   if(true == _hfpSiriProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         ::asf::dbus::DBusVariant variant;
         DbusVariant vValue;
         vValue.setBool(value);
         evolution::EvolutionGeniviDbusParser parser;
         parser.addVariantValue(variant, vValue);
         return proxy->sendSetPropertyRequest(*_callbackIf, property, variant);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusHfpSiriProxyIf implementation --- end

} //ccdbusif
