/**
 * @file EvolutionGeniviDbusHfpNetworkOperatorProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for HfpNetworkOperator.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for HfpNetworkOperator.
 */

#include "EvolutionGeniviDbusHfpNetworkOperatorProxyIf.h"
#include "IEvolutionGeniviDbusHfpNetworkOperatorCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::ofono::NetworkOperator;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpNetworkOperatorProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusHfpNetworkOperatorProxyIf::EvolutionGeniviDbusHfpNetworkOperatorProxyIf()
{
   // _hfpNetworkOperatorProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpNetworkOperatorProxyIf::EvolutionGeniviDbusHfpNetworkOperatorProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _hfpNetworkOperatorProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpNetworkOperatorProxyIf::~EvolutionGeniviDbusHfpNetworkOperatorProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): NetworkOperator"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpNetworkOperatorProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): NetworkOperator"));

   for(unsigned int i = 0; i < _hfpNetworkOperatorProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< NetworkOperatorProxy >& hfpNetworkOperatorProxy = _hfpNetworkOperatorProxy[i];
      if(NULL != hfpNetworkOperatorProxy.get())
      {
         hfpNetworkOperatorProxy->sendDeregisterAll();
      }
   }

   _hfpNetworkOperatorProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): NetworkOperator: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _hfpNetworkOperatorProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< NetworkOperatorProxy > hfpNetworkOperatorProxy = NetworkOperatorProxy::createProxy("evoHfpNetworkOperatorPort", parser.getInterface2BusName(evolution::IF_NETWORK_OPERATOR), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_NETWORK_OPERATOR)), *this);
         _hfpNetworkOperatorProxy.addProxyInstance(objPath, hfpNetworkOperatorProxy);
      }
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): NetworkOperator: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< NetworkOperatorProxy > hfpNetworkOperatorProxy;
   if(true == _hfpNetworkOperatorProxy.isProxyAvailable(hfpNetworkOperatorProxy, objPath))
   {
      hfpNetworkOperatorProxy->sendDeregisterAll();
   }
   _hfpNetworkOperatorProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::setCallbackIf(IEvolutionGeniviDbusHfpNetworkOperatorCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::operatorCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::operatorRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): NetworkOperator"));

   for(unsigned int i = 0; i < _hfpNetworkOperatorProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< NetworkOperatorProxy >& hfpNetworkOperatorProxy = _hfpNetworkOperatorProxy[i];
      if((NULL != hfpNetworkOperatorProxy.get()) && (proxy == hfpNetworkOperatorProxy))
      {
         if(NULL != _callbackIf)
         {
            hfpNetworkOperatorProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpNetworkOperatorProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpNetworkOperatorProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): NetworkOperator"));

   for(unsigned int i = 0; i < _hfpNetworkOperatorProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< NetworkOperatorProxy >& hfpNetworkOperatorProxy = _hfpNetworkOperatorProxy[i];
      if((NULL != hfpNetworkOperatorProxy.get()) && (proxy == hfpNetworkOperatorProxy))
      {
         hfpNetworkOperatorProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpNetworkOperatorProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusHfpNetworkOperatorProxyIf implementation --- start
act_t EvolutionGeniviDbusHfpNetworkOperatorProxyIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< NetworkOperatorProxy > proxy;
   if(true == _hfpNetworkOperatorProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusHfpNetworkOperatorProxyIf implementation --- end

} //ccdbusif
