/**
 * @file EvolutionGeniviDbusErrorCodes.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Properties for ErrorCodes.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Properties for ErrorCodes.
 */

#include "EvolutionGeniviDbusErrorCodes.h"
#include "FwStringUtils.h"
#include "FwAssert.h"

namespace ccdbusif {
namespace evolution {

EvolutionGeniviDbusErrorCodes::EvolutionGeniviDbusErrorCodes() : DbusProperties("ErrorCodes"),
   _workingStringCapacity(100),
   _workingString(),

   _org_bluez_Error_InvalidArguments("org.bluez.Error.InvalidArguments"),
   _org_bluez_Error_InProgress("org.bluez.Error.InProgress"),
   _org_bluez_Error_AlreadyExists("org.bluez.Error.AlreadyExists"),
   _org_bluez_Error_NotSupported("org.bluez.Error.NotSupported"),
   _org_bluez_Error_NotConnected("org.bluez.Error.NotConnected"),
   _org_bluez_Error_AlreadyConnected("org.bluez.Error.AlreadyConnected"),
   _org_bluez_Error_NotAvailable("org.bluez.Error.NotAvailable"),
   _org_bluez_Error_NotAuthorized("org.bluez.Error.NotAuthorized"),
   _org_bluez_Error_AgentNotAvailable("org.bluez.Error.AgentNotAvailable"),
   _org_bluez_Error_NotReady("org.bluez.Error.NotReady"),
   _org_bluez_Error_Failed("org.bluez.Error.Failed"),
   _org_ofono_Error_InvalidArguments("org.ofono.Error.InvalidArguments"),
   _org_ofono_Error_InvalidFormat("org.ofono.Error.InvalidFormat"),
   _org_ofono_Error_NotImplemented("org.ofono.Error.NotImplemented"),
   _org_ofono_Error_Failed("org.ofono.Error.Failed"),
   _org_ofono_Error_InProgress("org.ofono.Error.InProgress"),
   _org_ofono_Error_NotFound("org.ofono.Error.NotFound"),
   _org_ofono_Error_NotActive("org.ofono.Error.NotActive"),
   _org_ofono_Error_NotSupported("org.ofono.Error.NotSupported"),
   _org_ofono_Error_NotAvailable("org.ofono.Error.NotAvailable"),
   _org_ofono_Error_Timedout("org.ofono.Error.Timedout"),
   _org_ofono_Error_InUse("org.ofono.Error.InUse"),
   _org_bluez_obex_Error_InvalidArguments("org.bluez.obex.Error.InvalidArguments"),
   _org_bluez_obex_Error_Failed("org.bluez.obex.Error.Failed"),
   _org_bluez_obex_Error_Forbidden("org.bluez.obex.Error.Forbidden"),
   _org_bluez_obex_NotAuthorized("org.bluez.obex.NotAuthorized"),
   _org_bluez_obex_InProgress("org.bluez.obex.InProgress"),
   _org_bluez_obex_AlreadyConnected("org.bluez.obex.Error.AlreadyConnected"),
   _org_bluez_Error_ConnectionLost("org.bluez.Error.ConnectionLost"),
   _org_bluez_Error_AuthenticationError("org.bluez.Error.AuthenticationError"),
   _org_bluez_Error_PairingCancelled("org.bluez.Error.PairingCancelled"),
   _org_bluez_Error_MissingPINOrKey("org.bluez.Error.MissingPINOrKey"),
   _org_bluez_Error_UnknownReason("org.bluez.Error.UnknownReason"),
   _org_bluez_Error_DoesNotExist("org.bluez.Error.DoesNotExist"),
   _org_bluez_Error_DiscoverServicesCancelled("org.bluez.Error.DiscoverServicesCancelled"),
   _org_bluez_Error_ServiceIncoming("org.bluez.Error.ServiceIncoming"),
   _org_bluez_Error_ConnectionConflict("org.bluez.Error.ConnectCollision"),

   _org_freedesktop_Error_UnknownObject("org.freedesktop.DBus.Error.UnknownObject"),
   _org_freedesktop_Error_UnknownMethod("org.freedesktop.DBus.Error.UnknownMethod"),
   _org_freedesktop_Error_NoServer("org.freedesktop.DBus.Error.NoServer"),
   _org_freedesktop_Error_NoReply("org.freedesktop.DBus.Error.NoReply"),

   _net_connman_Error_Failed("net.connman.Error.Failed"),
   _net_connman_Error_InvalidArguments("net.connman.Error.InvalidArguments"),
   _net_connman_Error_PermissionDenied("net.connman.Error.PermissionDenied"),
   _net_connman_Error_PassphraseRequired("net.connman.Error.PassphraseRequired"),
   _net_connman_Error_NotRegistered("net.connman.Error.NotRegistered"),
   _net_connman_Error_NotUnique("net.connman.Error.NotUnique"),
   _net_connman_Error_NotSupported("net.connman.Error.NotSupported"),
   _net_connman_Error_NotImplemented("net.connman.Error.NotImplemented"),
   _net_connman_Error_NotFound("net.connman.Error.NotFound"),
   _net_connman_Error_NoCarrier("net.connman.Error.NoCarrier"),
   _net_connman_Error_InProgress("net.connman.Error.InProgress"),
   _net_connman_Error_AlreadyExists("net.connman.Error.AlreadyExists"),
   _net_connman_Error_AlreadyEnabled("net.connman.Error.AlreadyEnabled"),
   _net_connman_Error_AlreadyDisabled("net.connman.Error.AlreadyDisabled"),
   _net_connman_Error_AlreadyConnected("net.connman.Error.AlreadyConnected"),
   _net_connman_Error_NotConnected("net.connman.Error.NotConnected"),
   _net_connman_Error_OperationAborted("net.connman.Error.OperationAborted"),
   _net_connman_Error_OperationTimeout("net.connman.Error.OperationTimeout"),
   _net_connman_Error_InvalidService("net.connman.Error.InvalidService"),
   _net_connman_Error_InvalidProperty("net.connman.Error.InvalidProperty"),

   _subErrorCode_UnknownOpcode("unknown opcode"),
   _subErrorCode_IllegalFormat("illegal format"),
   _subErrorCode_IllegalParam("illegal param"),
   _subErrorCode_OperationFail("operation fail"),
   _subErrorCode_OperationFailed("Operation Failed"),
   _subErrorCode_SystemBusy("system busy"),
   _subErrorCode_IllegalOperation("illegal operation"),
   _subErrorCode_IllegalState("illegal state"),
   _subErrorCode_SdpFailure("sdp failure"),
   _subErrorCode_Abnormal("abnormal"),
   _subErrorCode_Aborted("aborted"),
   _subErrorCode_ExtendedError("extended error"),
   _subErrorCode_SdpConnectionError("sdp connection error"),
   _subErrorCode_TimeoutError("timeout error"),
   _subErrorCode_RestrictedConnection("restricted connection"),
   _subErrorCode_RawDataIllegal("rawdata illegal"),
   _subErrorCode_ServiceNotSupported("service notsupported"),
   _subErrorCode_MoreDataExpected("moredata expected"),
   _subErrorCode_SessionBusy("Session busy"),
   _subErrorCode_PageTimeout("page timeout"),
   _subErrorCode_RemoteReject("remote reject"),
   _subErrorCode_LmpResponseTimeout("LMP response timeout"),
   _subErrorCode_ConnectTimeout("connect timeout"),
   _subErrorCode_AuthenticationFailure("authentication failure"),
   _subErrorCode_RoleChangeNotAllowed("role change not allowed"),
   _subErrorCode_ConnectRejectBdAddr("connect reject bdaddr"),
   _subErrorCode_AclExists("ACL exists"),
   _subErrorCode_HwFailure("HW failure"),
   _subErrorCode_ConnectRejectLimitedResources("connect reject limited resources"),
   _subErrorCode_ConnectRejectSecurityReasons("connect reject security reasons"),
   _subErrorCode_RemoteDisconnectLowResource("remote disconnect low resource"),
   _subErrorCode_RemoteDisconnectPowerOff("remote disconnect power off"),
   _subErrorCode_PairingNotAllowed("pairing not allowed"),
   _subErrorCode_ConnectionAcceptTimeout("connection accept timeout"),
   _subErrorCode_LocalReject("local reject"),
   _subErrorCode_InterfaceRemoved("interface removed"),
   _subErrorCode_RemoteDisconnectAcl("remote disconnect acl"),
   _subErrorCode_PsFailure("ps failure"),
   _subErrorCode_Empty("")
{
   _workingString.reserve(_workingStringCapacity); // reserve a capacity of 100 bytes to avoid re-allocations during next steps

   _string2EnumList[convert2Lower(_org_bluez_Error_InvalidArguments)] = (int)ERROR_ORG_BLUEZ_ERROR_INVALIDARGUMENTS;
   _string2EnumList[convert2Lower(_org_bluez_Error_InProgress)] = (int)ERROR_ORG_BLUEZ_ERROR_INPROGRESS;
   _string2EnumList[convert2Lower(_org_bluez_Error_AlreadyExists)] = (int)ERROR_ORG_BLUEZ_ERROR_ALREADYEXISTS;
   _string2EnumList[convert2Lower(_org_bluez_Error_NotSupported)] = (int)ERROR_ORG_BLUEZ_ERROR_NOTSUPPORTED;
   _string2EnumList[convert2Lower(_org_bluez_Error_NotConnected)] = (int)ERROR_ORG_BLUEZ_ERROR_NOTCONNECTED;
   _string2EnumList[convert2Lower(_org_bluez_Error_AlreadyConnected)] = (int)ERROR_ORG_BLUEZ_ERROR_ALREADYCONNECTED;
   _string2EnumList[convert2Lower(_org_bluez_Error_NotAvailable)] = (int)ERROR_ORG_BLUEZ_ERROR_NOTAVAILABLE;
   _string2EnumList[convert2Lower(_org_bluez_Error_NotAuthorized)] = (int)ERROR_ORG_BLUEZ_ERROR_NOTAUTHORIZED;
   _string2EnumList[convert2Lower(_org_bluez_Error_AgentNotAvailable)] = (int)ERROR_ORG_BLUEZ_ERROR_AGENTNOTAVAILABLE;
   _string2EnumList[convert2Lower(_org_bluez_Error_NotReady)] = (int)ERROR_ORG_BLUEZ_ERROR_NOTREADY;
   _string2EnumList[convert2Lower(_org_bluez_Error_Failed)] = (int)ERROR_ORG_BLUEZ_ERROR_FAILED;
   _string2EnumList[convert2Lower(_org_ofono_Error_InvalidArguments)] = (int)ERROR_ORG_OFONO_ERROR_INVALIDARGUMENTS;
   _string2EnumList[convert2Lower(_org_ofono_Error_InvalidFormat)] = (int)ERROR_ORG_OFONO_ERROR_INVALIDFORMAT;
   _string2EnumList[convert2Lower(_org_ofono_Error_NotImplemented)] = (int)ERROR_ORG_OFONO_ERROR_NOTIMPLEMENTED;
   _string2EnumList[convert2Lower(_org_ofono_Error_Failed)] = (int)ERROR_ORG_OFONO_ERROR_FAILED;
   _string2EnumList[convert2Lower(_org_ofono_Error_InProgress)] = (int)ERROR_ORG_OFONO_ERROR_INPROGRESS;
   _string2EnumList[convert2Lower(_org_ofono_Error_NotFound)] = (int)ERROR_ORG_OFONO_ERROR_NOTFOUND;
   _string2EnumList[convert2Lower(_org_ofono_Error_NotActive)] = (int)ERROR_ORG_OFONO_ERROR_NOTACTIVE;
   _string2EnumList[convert2Lower(_org_ofono_Error_NotSupported)] = (int)ERROR_ORG_OFONO_ERROR_NOTSUPPORTED;
   _string2EnumList[convert2Lower(_org_ofono_Error_NotAvailable)] = (int)ERROR_ORG_OFONO_ERROR_NOTAVAILABLE;
   _string2EnumList[convert2Lower(_org_ofono_Error_Timedout)] = (int)ERROR_ORG_OFONO_ERROR_TIMEDOUT;
   _string2EnumList[convert2Lower(_org_ofono_Error_InUse)] = (int)ERROR_ORG_OFONO_ERROR_INUSE;
   _string2EnumList[convert2Lower(_org_bluez_obex_Error_InvalidArguments)] = (int)ERROR_ORG_BLUEZ_OBEX_ERROR_INVALIDARGUMENTS;
   _string2EnumList[convert2Lower(_org_bluez_obex_Error_Failed)] = (int)ERROR_ORG_BLUEZ_OBEX_ERROR_FAILED;
   _string2EnumList[convert2Lower(_org_bluez_obex_Error_Forbidden)] = (int)ERROR_ORG_BLUEZ_OBEX_ERROR_FORBIDDEN;
   _string2EnumList[convert2Lower(_org_bluez_obex_NotAuthorized)] = (int)ERROR_ORG_BLUEZ_OBEX_NOTAUTHORIZED;
   _string2EnumList[convert2Lower(_org_bluez_obex_InProgress)] = (int)ERROR_ORG_BLUEZ_OBEX_INPROGRESS;
   _string2EnumList[convert2Lower(_org_bluez_obex_AlreadyConnected)] = (int)ERROR_ORG_BLUEZ_OBEX_ALREADYCONNECTED;

   _string2EnumList[convert2Lower(_org_bluez_Error_ConnectionLost)] = (int)ERROR_ORG_BLUEZ_ERROR_CONNECTIONLOST;
   _string2EnumList[convert2Lower(_org_bluez_Error_AuthenticationError)] = (int)ERROR_ORG_BLUEZ_ERROR_AUTHENTICATIONERROR;
   _string2EnumList[convert2Lower(_org_bluez_Error_PairingCancelled)] = (int)ERROR_ORG_BLUEZ_ERROR_PAIRINGCANCELLED;
   _string2EnumList[convert2Lower(_org_bluez_Error_MissingPINOrKey)] = (int)ERROR_ORG_BLUEZ_ERROR_MISSINGPINORKEY;
   _string2EnumList[convert2Lower(_org_bluez_Error_UnknownReason)] = (int)ERROR_ORG_BLUEZ_ERROR_UNKNOWNREASON;
   _string2EnumList[convert2Lower(_org_bluez_Error_DoesNotExist)] = (int)ERROR_ORG_BLUEZ_ERROR_DOESNOTEXIST;
   _string2EnumList[convert2Lower(_org_bluez_Error_DiscoverServicesCancelled)] = (int)ERROR_ORG_BLUEZ_ERROR_DISCOVERSERVICESCANCELLED;
   _string2EnumList[convert2Lower(_org_bluez_Error_ServiceIncoming)] = (int)ERROR_ORG_BLUEZ_ERROR_SERVICEINCOMING;
   _string2EnumList[convert2Lower(_org_bluez_Error_ConnectionConflict)] = (int)ERROR_ORG_BLUEZ_ERROR_CONNECTIONCONFLICT;

   _string2EnumList[convert2Lower(_org_freedesktop_Error_UnknownObject)] = (int)ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNOBJECT;
   _string2EnumList[convert2Lower(_org_freedesktop_Error_UnknownMethod)] = (int)ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNMETHOD;
   _string2EnumList[convert2Lower(_org_freedesktop_Error_NoServer)] = (int)ERROR_ORG_FREEDESKTOP_ERROR_NOSERVER;
   _string2EnumList[convert2Lower(_org_freedesktop_Error_NoReply)] = (int)ERROR_ORG_FREEDESKTOP_ERROR_NOREPLY;

   _string2EnumList[convert2Lower(_net_connman_Error_Failed)] = (int)ERROR_NET_CONNMAN_ERROR_FAILED;
   _string2EnumList[convert2Lower(_net_connman_Error_InvalidArguments)] = (int)ERROR_NET_CONNMAN_ERROR_INVALIDARGUMENTS;
   _string2EnumList[convert2Lower(_net_connman_Error_PermissionDenied)] = (int)ERROR_NET_CONNMAN_ERROR_PERMISSIONDENIED;
   _string2EnumList[convert2Lower(_net_connman_Error_PassphraseRequired)] = (int)ERROR_NET_CONNMAN_ERROR_PASSPHRASEREQUIRED;
   _string2EnumList[convert2Lower(_net_connman_Error_NotRegistered)] = (int)ERROR_NET_CONNMAN_ERROR_NOTREGISTERED;
   _string2EnumList[convert2Lower(_net_connman_Error_NotUnique)] = (int)ERROR_NET_CONNMAN_ERROR_NOTUNIQUE;
   _string2EnumList[convert2Lower(_net_connman_Error_NotSupported)] = (int)ERROR_NET_CONNMAN_ERROR_NOTSUPPORTED;
   _string2EnumList[convert2Lower(_net_connman_Error_NotImplemented)] = (int)ERROR_NET_CONNMAN_ERROR_NOTIMPLEMENTED;
   _string2EnumList[convert2Lower(_net_connman_Error_NotFound)] = (int)ERROR_NET_CONNMAN_ERROR_NOTFOUND;
   _string2EnumList[convert2Lower(_net_connman_Error_NoCarrier)] = (int)ERROR_NET_CONNMAN_ERROR_NOCARRIER;
   _string2EnumList[convert2Lower(_net_connman_Error_InProgress)] = (int)ERROR_NET_CONNMAN_ERROR_INPROGRESS;
   _string2EnumList[convert2Lower(_net_connman_Error_AlreadyExists)] = (int)ERROR_NET_CONNMAN_ERROR_ALREADYEXISTS;
   _string2EnumList[convert2Lower(_net_connman_Error_AlreadyEnabled)] = (int)ERROR_NET_CONNMAN_ERROR_ALREADYENABLED;
   _string2EnumList[convert2Lower(_net_connman_Error_AlreadyDisabled)] = (int)ERROR_NET_CONNMAN_ERROR_ALREADYDISABLED;
   _string2EnumList[convert2Lower(_net_connman_Error_AlreadyConnected)] = (int)ERROR_NET_CONNMAN_ERROR_ALREADYCONNECTED;
   _string2EnumList[convert2Lower(_net_connman_Error_NotConnected)] = (int)ERROR_NET_CONNMAN_ERROR_NOTCONNECTED;
   _string2EnumList[convert2Lower(_net_connman_Error_OperationAborted)] = (int)ERROR_NET_CONNMAN_ERROR_OPERATIONABORTED;
   _string2EnumList[convert2Lower(_net_connman_Error_OperationTimeout)] = (int)ERROR_NET_CONNMAN_ERROR_OPERATIONTIMEOUT;
   _string2EnumList[convert2Lower(_net_connman_Error_InvalidService)] = (int)ERROR_NET_CONNMAN_ERROR_INVALIDSERVICE;
   _string2EnumList[convert2Lower(_net_connman_Error_InvalidProperty)] = (int)ERROR_NET_CONNMAN_ERROR_INVALIDPROPERTY;

   _string2EnumList[convert2Lower(_subErrorCode_UnknownOpcode)] = (int)ERROR_UNKNOWN_OPCODE;
   _string2EnumList[convert2Lower(_subErrorCode_IllegalFormat)] = (int)ERROR_ILLEGAL_FORMAT;
   _string2EnumList[convert2Lower(_subErrorCode_IllegalParam)] = (int)ERROR_ILLEGAL_PARAM;
   _string2EnumList[convert2Lower(_subErrorCode_OperationFail)] = (int)ERROR_OPERATION_FAIL;
   _string2EnumList[convert2Lower(_subErrorCode_OperationFailed)] = (int)ERROR_OPERATION_FAIL;
   _string2EnumList[convert2Lower(_subErrorCode_SystemBusy)] = (int)ERROR_SYSTEM_BUSY;
   _string2EnumList[convert2Lower(_subErrorCode_IllegalOperation)] = (int)ERROR_ILLEGAL_OPERATION;
   _string2EnumList[convert2Lower(_subErrorCode_IllegalState)] = (int)ERROR_ILLEGAL_STATE;
   _string2EnumList[convert2Lower(_subErrorCode_SdpFailure)] = (int)ERROR_SDP_FAILURE;
   _string2EnumList[convert2Lower(_subErrorCode_Abnormal)] = (int)ERROR_ABNORMAL;
   _string2EnumList[convert2Lower(_subErrorCode_Aborted)] = (int)ERROR_ABORTED;
   _string2EnumList[convert2Lower(_subErrorCode_ExtendedError)] = (int)ERROR_EXTENDED_ERROR;
   _string2EnumList[convert2Lower(_subErrorCode_SdpConnectionError)] = (int)ERROR_SDP_CONNECTION_ERROR;
   _string2EnumList[convert2Lower(_subErrorCode_TimeoutError)] = (int)ERROR_TIMEOUT_ERROR;
   _string2EnumList[convert2Lower(_subErrorCode_RestrictedConnection)] = (int)ERROR_RESTRICTED_CONNECTION;
   _string2EnumList[convert2Lower(_subErrorCode_RawDataIllegal)] = (int)ERROR_RAW_DATA_ILLEGAL;
   _string2EnumList[convert2Lower(_subErrorCode_ServiceNotSupported)] = (int)ERROR_SERVICE_NOT_SUPPORTED;
   _string2EnumList[convert2Lower(_subErrorCode_MoreDataExpected)] = (int)ERROR_MORE_DATA_EXPECTED;
   _string2EnumList[convert2Lower(_subErrorCode_PageTimeout)] = (int)ERROR_PAGE_TIMEOUT;
   _string2EnumList[convert2Lower(_subErrorCode_SessionBusy)] = (int)ERROR_SESSION_BUSY;
   _string2EnumList[convert2Lower(_subErrorCode_RemoteReject)] = (int)ERROR_REMOTE_REJECT;
   _string2EnumList[convert2Lower(_subErrorCode_LmpResponseTimeout)] = (int)ERROR_LMP_RESPONSE_TIMEOUT;
   _string2EnumList[convert2Lower(_subErrorCode_ConnectTimeout)] = (int)ERROR_CONNECT_TIMEOUT;
   _string2EnumList[convert2Lower(_subErrorCode_AuthenticationFailure)] = (int)ERROR_AUTHENTICATION_FAILURE;
   _string2EnumList[convert2Lower(_subErrorCode_RoleChangeNotAllowed)] = (int)ERROR_ROLE_CHANGE_NOT_ALLOWED;
   _string2EnumList[convert2Lower(_subErrorCode_ConnectRejectBdAddr)] = (int)ERROR_CONNECT_REJECT_BDADDR;
   _string2EnumList[convert2Lower(_subErrorCode_AclExists)] = (int)ERROR_ACL_EXISTS;
   _string2EnumList[convert2Lower(_subErrorCode_HwFailure)] = (int)ERROR_HW_FAILURE;
   _string2EnumList[convert2Lower(_subErrorCode_ConnectRejectLimitedResources)] = (int)ERROR_CONNECT_REJECT_LIMITED_RESOURCES;
   _string2EnumList[convert2Lower(_subErrorCode_ConnectRejectSecurityReasons)] = (int)ERROR_CONNECT_REJECT_SECURITY_REASONS;
   _string2EnumList[convert2Lower(_subErrorCode_RemoteDisconnectLowResource)] = (int)ERROR_REMOTE_DISCONNECT_LOW_RESOURCE;
   _string2EnumList[convert2Lower(_subErrorCode_RemoteDisconnectPowerOff)] = (int)ERROR_REMOTE_DISCONNECT_POWER_OFF;
   _string2EnumList[convert2Lower(_subErrorCode_PairingNotAllowed)] = (int)ERROR_PAIRING_NOT_ALLOWED;
   _string2EnumList[convert2Lower(_subErrorCode_ConnectionAcceptTimeout)] = (int)ERROR_CONNECTION_ACCEPT_TIMEOUT;
   _string2EnumList[convert2Lower(_subErrorCode_LocalReject)] = (int)ERROR_LOCAL_REJECT;
   _string2EnumList[convert2Lower(_subErrorCode_InterfaceRemoved)] = (int)ERROR_INTERFACE_REMOVED;
   _string2EnumList[convert2Lower(_subErrorCode_RemoteDisconnectAcl)] = (int)ERROR_REMOTE_DISCONNECT_ACL;
   _string2EnumList[convert2Lower(_subErrorCode_PsFailure)] = (int)ERROR_PS_FAILURE;
   _string2EnumList[convert2Lower(_subErrorCode_Empty)] = (int)ERROR_EMPTY;

   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_INVALIDARGUMENTS] = &_org_bluez_Error_InvalidArguments;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_INPROGRESS] = &_org_bluez_Error_InProgress;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_ALREADYEXISTS] = &_org_bluez_Error_AlreadyExists;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_NOTSUPPORTED] = &_org_bluez_Error_NotSupported;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_NOTCONNECTED] = &_org_bluez_Error_NotConnected;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_ALREADYCONNECTED] = &_org_bluez_Error_AlreadyConnected;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_NOTAVAILABLE] = &_org_bluez_Error_NotAvailable;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_NOTAUTHORIZED] = &_org_bluez_Error_NotAuthorized;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_AGENTNOTAVAILABLE] = &_org_bluez_Error_AgentNotAvailable;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_NOTREADY] = &_org_bluez_Error_NotReady;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_FAILED] = &_org_bluez_Error_Failed;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_INVALIDARGUMENTS] = &_org_ofono_Error_InvalidArguments;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_INVALIDFORMAT] = &_org_ofono_Error_InvalidFormat;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_NOTIMPLEMENTED] = &_org_ofono_Error_NotImplemented;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_FAILED] = &_org_ofono_Error_Failed;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_INPROGRESS] = &_org_ofono_Error_InProgress;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_NOTFOUND] = &_org_ofono_Error_NotFound;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_NOTACTIVE] = &_org_ofono_Error_NotActive;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_NOTSUPPORTED] = &_org_ofono_Error_NotSupported;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_NOTAVAILABLE] = &_org_ofono_Error_NotAvailable;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_TIMEDOUT] = &_org_ofono_Error_Timedout;
   _enum2StringList[(int)ERROR_ORG_OFONO_ERROR_INUSE] = &_org_ofono_Error_InUse;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_ERROR_INVALIDARGUMENTS] = &_org_bluez_obex_Error_InvalidArguments;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_ERROR_FAILED] = &_org_bluez_obex_Error_Failed;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_ERROR_FORBIDDEN] = &_org_bluez_obex_Error_Forbidden;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_NOTAUTHORIZED] = &_org_bluez_obex_NotAuthorized;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_INPROGRESS] = &_org_bluez_obex_InProgress;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_OBEX_ALREADYCONNECTED] = &_org_bluez_obex_AlreadyConnected;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_CONNECTIONLOST] = &_org_bluez_Error_ConnectionLost;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_AUTHENTICATIONERROR] = &_org_bluez_Error_AuthenticationError;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_PAIRINGCANCELLED] = &_org_bluez_Error_PairingCancelled;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_MISSINGPINORKEY] = &_org_bluez_Error_MissingPINOrKey;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_UNKNOWNREASON] = &_org_bluez_Error_UnknownReason;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_DOESNOTEXIST] = &_org_bluez_Error_DoesNotExist;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_DISCOVERSERVICESCANCELLED] = &_org_bluez_Error_DiscoverServicesCancelled;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_SERVICEINCOMING] = &_org_bluez_Error_ServiceIncoming;
   _enum2StringList[(int)ERROR_ORG_BLUEZ_ERROR_CONNECTIONCONFLICT] = &_org_bluez_Error_ConnectionConflict;

   _enum2StringList[(int)ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNOBJECT] = &_org_freedesktop_Error_UnknownObject;
   _enum2StringList[(int)ERROR_ORG_FREEDESKTOP_ERROR_UNKNOWNMETHOD] = &_org_freedesktop_Error_UnknownMethod;
   _enum2StringList[(int)ERROR_ORG_FREEDESKTOP_ERROR_NOSERVER] = &_org_freedesktop_Error_NoServer;
   _enum2StringList[(int)ERROR_ORG_FREEDESKTOP_ERROR_NOREPLY] = &_org_freedesktop_Error_NoReply;

   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_FAILED] = &_net_connman_Error_Failed;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_INVALIDARGUMENTS] = &_net_connman_Error_InvalidArguments;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_PERMISSIONDENIED] = &_net_connman_Error_PermissionDenied;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_PASSPHRASEREQUIRED] = &_net_connman_Error_PassphraseRequired;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTREGISTERED] = &_net_connman_Error_NotRegistered;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTUNIQUE] = &_net_connman_Error_NotUnique;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTSUPPORTED] = &_net_connman_Error_NotSupported;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTIMPLEMENTED] = &_net_connman_Error_NotImplemented;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTFOUND] = &_net_connman_Error_NotFound;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOCARRIER] = &_net_connman_Error_NoCarrier;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_INPROGRESS] = &_net_connman_Error_InProgress;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_ALREADYEXISTS] = &_net_connman_Error_AlreadyExists;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_ALREADYENABLED] = &_net_connman_Error_AlreadyEnabled;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_ALREADYDISABLED] = &_net_connman_Error_AlreadyDisabled;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_ALREADYCONNECTED] = &_net_connman_Error_AlreadyConnected;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_NOTCONNECTED] = &_net_connman_Error_NotConnected;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_OPERATIONABORTED] = &_net_connman_Error_OperationAborted;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_OPERATIONTIMEOUT] = &_net_connman_Error_OperationTimeout;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_INVALIDSERVICE] = &_net_connman_Error_InvalidService;
   _enum2StringList[(int)ERROR_NET_CONNMAN_ERROR_INVALIDPROPERTY] = &_net_connman_Error_InvalidProperty;

   _enum2StringList[(int)ERROR_UNKNOWN_OPCODE] = &_subErrorCode_UnknownOpcode;
   _enum2StringList[(int)ERROR_ILLEGAL_FORMAT] = &_subErrorCode_IllegalFormat;
   _enum2StringList[(int)ERROR_ILLEGAL_PARAM] = &_subErrorCode_IllegalParam;
   _enum2StringList[(int)ERROR_OPERATION_FAIL] = &_subErrorCode_OperationFail;
   _enum2StringList[(int)ERROR_SYSTEM_BUSY] = &_subErrorCode_SystemBusy;
   _enum2StringList[(int)ERROR_ILLEGAL_OPERATION] = &_subErrorCode_IllegalOperation;
   _enum2StringList[(int)ERROR_ILLEGAL_STATE] = &_subErrorCode_IllegalState;
   _enum2StringList[(int)ERROR_SDP_FAILURE] = &_subErrorCode_SdpFailure;
   _enum2StringList[(int)ERROR_ABNORMAL] = &_subErrorCode_Abnormal;
   _enum2StringList[(int)ERROR_ABORTED] = &_subErrorCode_Aborted;
   _enum2StringList[(int)ERROR_EXTENDED_ERROR] = &_subErrorCode_ExtendedError;
   _enum2StringList[(int)ERROR_SDP_CONNECTION_ERROR] = &_subErrorCode_SdpConnectionError;
   _enum2StringList[(int)ERROR_TIMEOUT_ERROR] = &_subErrorCode_TimeoutError;
   _enum2StringList[(int)ERROR_RESTRICTED_CONNECTION] = &_subErrorCode_RestrictedConnection;
   _enum2StringList[(int)ERROR_RAW_DATA_ILLEGAL] = &_subErrorCode_RawDataIllegal;
   _enum2StringList[(int)ERROR_SERVICE_NOT_SUPPORTED] = &_subErrorCode_ServiceNotSupported;
   _enum2StringList[(int)ERROR_MORE_DATA_EXPECTED] = &_subErrorCode_MoreDataExpected;
   _enum2StringList[(int)ERROR_PAGE_TIMEOUT] = &_subErrorCode_PageTimeout;
   _enum2StringList[(int)ERROR_SESSION_BUSY] = &_subErrorCode_SessionBusy;
   _enum2StringList[(int)ERROR_REMOTE_REJECT] = &_subErrorCode_RemoteReject;
   _enum2StringList[(int)ERROR_LMP_RESPONSE_TIMEOUT] = &_subErrorCode_LmpResponseTimeout;
   _enum2StringList[(int)ERROR_CONNECT_TIMEOUT] = &_subErrorCode_ConnectTimeout;
   _enum2StringList[(int)ERROR_AUTHENTICATION_FAILURE] = &_subErrorCode_AuthenticationFailure;
   _enum2StringList[(int)ERROR_ROLE_CHANGE_NOT_ALLOWED] = &_subErrorCode_RoleChangeNotAllowed;
   _enum2StringList[(int)ERROR_CONNECT_REJECT_BDADDR] = &_subErrorCode_ConnectRejectBdAddr;
   _enum2StringList[(int)ERROR_ACL_EXISTS] = &_subErrorCode_AclExists;
   _enum2StringList[(int)ERROR_HW_FAILURE] = &_subErrorCode_HwFailure;
   _enum2StringList[(int)ERROR_CONNECT_REJECT_LIMITED_RESOURCES] = &_subErrorCode_ConnectRejectLimitedResources;
   _enum2StringList[(int)ERROR_CONNECT_REJECT_SECURITY_REASONS] = &_subErrorCode_ConnectRejectSecurityReasons;
   _enum2StringList[(int)ERROR_REMOTE_DISCONNECT_LOW_RESOURCE] = &_subErrorCode_RemoteDisconnectLowResource;
   _enum2StringList[(int)ERROR_REMOTE_DISCONNECT_POWER_OFF] = &_subErrorCode_RemoteDisconnectPowerOff;
   _enum2StringList[(int)ERROR_PAIRING_NOT_ALLOWED] = &_subErrorCode_PairingNotAllowed;
   _enum2StringList[(int)ERROR_CONNECTION_ACCEPT_TIMEOUT] = &_subErrorCode_ConnectionAcceptTimeout;
   _enum2StringList[(int)ERROR_LOCAL_REJECT] = &_subErrorCode_LocalReject;
   _enum2StringList[(int)ERROR_INTERFACE_REMOVED] = &_subErrorCode_InterfaceRemoved;
   _enum2StringList[(int)ERROR_REMOTE_DISCONNECT_ACL] = &_subErrorCode_RemoteDisconnectAcl;
   _enum2StringList[(int)ERROR_PS_FAILURE] = &_subErrorCode_PsFailure;
   _enum2StringList[(int)ERROR_EMPTY] = &_subErrorCode_Empty;
}

EvolutionGeniviDbusErrorCodes::~EvolutionGeniviDbusErrorCodes()
{
}

ErrorCode EvolutionGeniviDbusErrorCodes::getErrorCode2Enum(const ::std::string& value) const
{
   ::std::string tmpString(value);
   ::fw::convertString2LowerCase(tmpString);

   return (ErrorCode)convertString2Enum(tmpString);
}

const ::std::string& EvolutionGeniviDbusErrorCodes::getErrorCode2String(const ErrorCode value) const
{
   return convertEnum2String((const int)value);
}

const ::std::string& EvolutionGeniviDbusErrorCodes::getErrorCode2StringTrace(const ErrorCode value) const
{
   if(ERROR_UNKNOWN <= value)
   {
      return _unknownString;
   }

   const ::std::string& result = convertEnum2String((const int)value, false);

   if(_dummyString == result)
   {
      return _unknownString;
   }

   return result;
}

ErrorCode EvolutionGeniviDbusErrorCodes::getErrorCode2Enum(const ::std::string& errorName, const ::std::string& errorMessage) const
{
   // 1. pass
   const ErrorCode returnValue = getErrorCode2Enum(errorName);

   // 2. pass
   if((ERROR_ORG_BLUEZ_ERROR_FAILED == returnValue) || (ERROR_ORG_BLUEZ_OBEX_ERROR_FAILED == returnValue))
   {
      const ErrorCode returnValue2 = getErrorCode2Enum(errorMessage);

      if((ERROR_UNKNOWN > returnValue2) && (ERROR_EMPTY != returnValue2))
      {
         return returnValue2;
      }
   }

   return returnValue;
}

const ::std::string& EvolutionGeniviDbusErrorCodes::convert2Lower(const ::std::string& inputValue)
{
   _workingString = inputValue;

   // FW_NORMAL_ASSERT(_workingStringCapacity == _workingString.capacity());

   ::fw::convertString2LowerCase(_workingString);

   return _workingString;
}

} //evolution
} //ccdbusif
