/**
 * @file EvolutionGeniviDbusDeviceProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for Device.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for Device.
 */

#include "EvolutionGeniviDbusDeviceProxyIf.h"
#include "IEvolutionGeniviDbusDeviceCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::Device1;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusDeviceProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusDeviceProxyIf::EvolutionGeniviDbusDeviceProxyIf()
{
   // _deviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDeviceProxyIf::EvolutionGeniviDbusDeviceProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _deviceProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDeviceProxyIf::~EvolutionGeniviDbusDeviceProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusDeviceProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusDeviceProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Device1"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _deviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusDeviceProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Device1"));

   for(unsigned int i = 0; i < _deviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< Device1Proxy >& deviceProxy = _deviceProxy[i];
      if(NULL != deviceProxy.get())
      {
         deviceProxy->sendDeregisterAll();
      }
   }

   _deviceProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusDeviceProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Device1: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _deviceProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< Device1Proxy > deviceProxy = Device1Proxy::createProxy("evoDevicePort", parser.getInterface2BusName(evolution::IF_DEVICE), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_DEVICE)), *this);
         _deviceProxy.addProxyInstance(objPath, deviceProxy);
      }
   }
}

void EvolutionGeniviDbusDeviceProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Device1: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< Device1Proxy > deviceProxy;
   if(true == _deviceProxy.isProxyAvailable(deviceProxy, objPath))
   {
      deviceProxy->sendDeregisterAll();
   }
   _deviceProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusDeviceProxyIf::setCallbackIf(IEvolutionGeniviDbusDeviceCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusDeviceProxyIf::deviceCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusDeviceProxyIf::deviceRemoved(const ::std::string& objPath, const bool destroyDirectly /*= false*/)
{
   // destroy runtime proxy
   if (true == destroyDirectly)
   {
      // Called within context of ASF component thread, we can directly destroy the proxy.
      destroyRuntimeProxy(objPath);
   }
   else
   {
      if ((NULL != _callbackIf) && (NULL != _controllerClient))
      {
         _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
      }
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusDeviceProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Device1"));

   for(unsigned int i = 0; i < _deviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< Device1Proxy >& deviceProxy = _deviceProxy[i];
      if((NULL != deviceProxy.get()) && (proxy == deviceProxy))
      {
         if(NULL != _callbackIf)
         {
            deviceProxy->sendReportLinkkeyRegister(*_callbackIf);
            deviceProxy->sendIncomingPairingCompRegister(*_callbackIf);
            deviceProxy->sendAddressRegister(*_callbackIf);
            deviceProxy->sendClassRegister(*_callbackIf);
            deviceProxy->sendIconRegister(*_callbackIf);
            deviceProxy->sendUUIDsRegister(*_callbackIf);
            deviceProxy->sendPairedRegister(*_callbackIf);
            deviceProxy->sendConnectedRegister(*_callbackIf);
            deviceProxy->sendTrustedRegister(*_callbackIf);
            deviceProxy->sendBlockedRegister(*_callbackIf);
            deviceProxy->sendAliasRegister(*_callbackIf);
            deviceProxy->sendAdapterRegister(*_callbackIf);
            deviceProxy->sendModaliasRegister(*_callbackIf);
            deviceProxy->sendRSSIRegister(*_callbackIf);
            deviceProxy->sendVendorRegister(*_callbackIf);
            deviceProxy->sendVendorSourceRegister(*_callbackIf);
            deviceProxy->sendProductRegister(*_callbackIf);
            deviceProxy->sendMapInfoRegister(*_callbackIf);
            deviceProxy->sendConnectedServiceRegister(*_callbackIf);
            deviceProxy->sendVersionRegister(*_callbackIf);
            deviceProxy->sendPBAPInfoRegister(*_callbackIf);
            deviceProxy->sendModeRegister(*_callbackIf);
            deviceProxy->sendRoleRegister(*_callbackIf);
            deviceProxy->sendSDPProcessReportRegister(*_callbackIf);
            deviceProxy->sendEnableAvpPauseRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), deviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusDeviceProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Device1"));

   for(unsigned int i = 0; i < _deviceProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< Device1Proxy >& deviceProxy = _deviceProxy[i];
      if((NULL != deviceProxy.get()) && (proxy == deviceProxy))
      {
         deviceProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), deviceProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEvolutionGeniviDbusDeviceProxyIf implementation --- start
act_t EvolutionGeniviDbusDeviceProxyIf::sendConnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendConnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendDisconnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDisconnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendPairRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendPairRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendCancelPairingRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendCancelPairingRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendDiscoverServicesRequest(const ::std::string& objPath, const ::std::string& pattern)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDiscoverServicesRequest(*_callbackIf, pattern);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendCancelDiscoveryRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendCancelDiscoveryRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendAddressGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAddressGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendClassGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendClassGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendIconGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendIconGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendUUIDsGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendUUIDsGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendPairedGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendPairedGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendConnectedGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendConnectedGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendTrustedGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendTrustedGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

void EvolutionGeniviDbusDeviceProxyIf::sendTrustedSet(const ::std::string& objPath, bool trusted)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         proxy->sendTrustedSet(trusted);
      }
   }
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendBlockedGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendBlockedGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

void EvolutionGeniviDbusDeviceProxyIf::sendBlockedSet(const ::std::string& objPath, bool blocked)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         proxy->sendBlockedSet(blocked);
      }
   }
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendAliasGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAliasGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

void EvolutionGeniviDbusDeviceProxyIf::sendAliasSet(const ::std::string& objPath, const ::std::string& alias)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         proxy->sendAliasSet(alias);
      }
   }
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendAdapterGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAdapterGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendModaliasGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendModaliasGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendRSSIGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendRSSIGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendVendorGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendVendorGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendVendorSourceGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendVendorSourceGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendProductGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendProductGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendMapInfoGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendMapInfoGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendConnectedServiceGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendConnectedServiceGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendVersionGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendVersionGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendPBAPInfoGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendPBAPInfoGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendModeGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendModeGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendRoleGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendRoleGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusDeviceProxyIf::sendEnableAvpPauseGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendEnableAvpPauseGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

void EvolutionGeniviDbusDeviceProxyIf::sendEnableAvpPauseSet(const ::std::string& objPath, bool enableAvpPause)
{
   ::boost::shared_ptr< Device1Proxy > proxy;
   if(true == _deviceProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         proxy->sendEnableAvpPauseSet(enableAvpPause);
      }
   }
}
// IEvolutionGeniviDbusDeviceProxyIf implementation --- end

} //ccdbusif
