/**
 * @file IEvolutionGeniviDbusSppSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Spp.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Spp.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_SPP_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_SPP_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <vector>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusSppCallbackIf;

/**
 * Send request interface class for Spp.
 */
class IEvolutionGeniviDbusSppSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusSppSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusSppCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send BtApplSppDataReq Request.
    * This method is used to send outgoing data to the remote device via the SPP service
    *
    * Possible Errors:
    *    org.bluez.Error InvalidArguments
    *    org.bluez.Error.Failed
    *
    * @param[in] instance - Instance index of SPP,please set the special instance value that SPP connected.
    * @param[in] data - The data to transmit (Length of the data parameter in Range:0x0001-0x0200)
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplSppDataReqRequest(uint8 instance, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send BtApplSppSetUuidReq Request.
    *
    * This method is used to register an init UUID and a receive UUID value for SPP service
    *
    * Possible Errors:
    *    org.bluez.Error InvalidArguments
    *    org.bluez.Error.Failed
    *
    * @param[in] instance - Instance index of SPP ,please set the special instance value that SPP connected.
    * @param[in] data - The data to set (Length of the data in Range:0x0006-0x0022)
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplSppSetUuidReqRequest(uint8 instance, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send BtApplSppFlowCtrlReq Request.
    *
    * This method is used to control the data transfer.
    *
    * Possible Errors:
    *    org.bluez.Error InvalidArguments
    *    org.bluez.Error.Failed
    *
    * @param[in] instance - Instance index of SPP ,please set the special instance value that SPP connected.
    * @param[in] action - The action of flow control state 0x00 – go, 0x01 – stop
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplSppFlowCtrlReqRequest(uint8 instance, uint8 action) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_SPP_SEND_REQUEST_IF_H_
