/**
 * @file IEvolutionGeniviDbusObexTransferSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for ObexTransfer.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for ObexTransfer.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_OBEX_TRANSFER_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_OBEX_TRANSFER_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusObexTransferCallbackIf;

/**
 * Send request interface class for ObexTransfer.
 */
class IEvolutionGeniviDbusObexTransferSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusObexTransferSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusObexTransferCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created transfer.
    *
    * @param[in] objPath: object path of created transfer
    */
   virtual void transferCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed transfer.
    *
    * @param[in] objPath: object path of removed transfer
    */
   virtual void transferRemoved(const ::std::string& objPath) = 0;

   /**
    * Send Cancel Request.
    * Stops the current transference.
    *
    * Possible error:
    *    org.bluez.obex.Error.NotAuthorized
    *    org.bluez.obex.Error.Inprogress
    *    org.bluez.obex.Error.Failed
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCancelRequest(const ::std::string& objPath) = 0;

   /**
    * Send Status Get.
    * Request the value of the property 'Status' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStatusGet(const ::std::string& objPath) = 0;

   /**
    * Send Session Get.
    * Request the value of the property 'Session' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSessionGet(const ::std::string& objPath) = 0;

   /**
    * Send Name Get.
    * Request the value of the property 'Name' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendNameGet(const ::std::string& objPath) = 0;

   /**
    * Send Type Get.
    * Request the value of the property 'Type' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTypeGet(const ::std::string& objPath) = 0;

   /**
    * Send Time Get.
    * Request the value of the property 'Time' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTimeGet(const ::std::string& objPath) = 0;

   /**
    * Send Size Get.
    * Request the value of the property 'Size' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSizeGet(const ::std::string& objPath) = 0;

   /**
    * Send Transferred Get.
    * Request the value of the property 'Transferred' from the D-Bus server.
    * This method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTransferredGet(const ::std::string& objPath) = 0;

   /**
    * Send Filename Get.
    * Request the value of the property 'Filename' from the D-Bus server.
    * This method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of transfer
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendFilenameGet(const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_OBEX_TRANSFER_SEND_REQUEST_IF_H_
