/**
 * @file IEvolutionGeniviDbusObexSessionSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for ObexSession.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for ObexSession.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_OBEX_SESSION_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_OBEX_SESSION_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusObexSessionCallbackIf;

/**
 * Send request interface class for ObexSession.
 */
class IEvolutionGeniviDbusObexSessionSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusObexSessionSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusObexSessionCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created session.
    *
    * @param[in] objPath: object path of created session
    */
   virtual void sessionCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed session.
    *
    * @param[in] objPath: object path of removed session
    */
   virtual void sessionRemoved(const ::std::string& objPath) = 0;

   /**
    * Send Source Get.
    * Request the value of the property 'Source' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of session
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSourceGet(const ::std::string& objPath) = 0;

   /**
    * Send Destination Get.
    * Request the value of the property 'Destination' from the D-Bus server
    *
    * @param[in] objPath: object path of session
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDestinationGet(const ::std::string& objPath) = 0;

   /**
    * Send Channel Get.
    * Request the value of the property 'Channel' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of session
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendChannelGet(const ::std::string& objPath) = 0;

   /**
    * Send Target Get.
    * Request the value of the property 'Target' from the D-Bus server. This
    * method sends a Get message to the D-Bus server.
    *
    * @param[in] objPath: object path of session
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTargetGet(const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_OBEX_SESSION_SEND_REQUEST_IF_H_
