/**
 * @file IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for HfpVoiceCallManager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for HfpVoiceCallManager.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_HFP_VOICE_CALL_MANAGER_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_HFP_VOICE_CALL_MANAGER_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusHfpVoiceCallManagerCallbackIf;

/**
 * Send request interface class for HfpVoiceCallManager.
 */
class IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusHfpVoiceCallManagerSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpVoiceCallManagerCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    */
   virtual void deviceRemoved(const ::std::string& objPath) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send GetCalls Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetCallsRequest(const ::std::string& objPath) = 0;

   /**
    * Send Dial Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] number: number to be dialed
    * @param[in] hide_callerid: caller id state
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDialRequest(const ::std::string& objPath, const ::std::string& number, const ::std::string& hide_callerid) = 0;

   /**
    * Send MemDial Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] numberindex: index number of the dial request
    * @param[in] hide_callerid: caller id state
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendMemDialRequest(const ::std::string& objPath, uint16 numberindex, const ::std::string& hide_callerid) = 0;

   /**
    * Send Redial Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] hide_callerid: caller id state
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRedialRequest(const ::std::string& objPath, const ::std::string& hide_callerid) = 0;

   /**
    * Send SwapCalls Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSwapCallsRequest(const ::std::string& objPath) = 0;

   /**
    * Send ReleaseAndAnswer Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendReleaseAndAnswerRequest(const ::std::string& objPath) = 0;

   /**
    * Send ReleaseAndSwap Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendReleaseAndSwapRequest(const ::std::string& objPath) = 0;

   /**
    * Send HoldAndAnswer Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendHoldAndAnswerRequest(const ::std::string& objPath) = 0;

   /**
    * Send HangupAll Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendHangupAllRequest(const ::std::string& objPath) = 0;

   /**
    * Send PrivateChat Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] call: call name to which private chat request has to be sent
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPrivateChatRequest(const ::std::string& objPath, const ::std::string& call) = 0;

   /**
    * Send CreateMultiparty Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCreateMultipartyRequest(const ::std::string& objPath) = 0;

   /**
    * Send HangupMultiparty Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendHangupMultipartyRequest(const ::std::string& objPath) = 0;

   /**
    * Send SendTones Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] sendTones: sendTone name
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSendTonesRequest(const ::std::string& objPath, const ::std::string& sendTones) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_HFP_VOICE_CALL_MANAGER_SEND_REQUEST_IF_H_
